/**********************************************************************
                                                                     
   Copyright (C) 2002,2003 Emil D. Kohn   emild@cs.technion.ac.il     
                                                                     
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 **********************************************************************/



#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "defs.h"

#define NO_MAIN


#define CRC_GENERATOR       0x04C11DB7 /* x^32 + x^26 + x^23 + x^22 + x^16 + x^12 + x^11 + 
				          x^10 + x^8  + x^7  + x^5  + x^4  + x^4  + 
					  x^2 + x + 1 (ITU rec. I.363.5) 
					  MSB is implicitly set to 1 */
#define HEC_GENERATOR       0x07       /* x^8 + x^2 + x + 1   (ITU rec. I.432) 
					  MSB is implicitly 1 */


#define MAKE_BYTE(hi,lo) (((hi) << 4) | (lo))


static u32 crc32_1(int nbits)
{
  int i;
  u32 result = 0xFFFFFFFF;

  assert(nbits >= 0);

  for (i=0; i<nbits; i++) {
    int msb = (result & 0x80000000);

    result <<= 1;

    if (msb) 
      result ^= CRC_GENERATOR;
  }
  return result;
}



static u32 crc32_2(const u8 data[], int nbytes)
{
  int i,j,msb=0;
  u32 result = 0;
 
  for (i=0; i < nbytes+4; i++) {
    u8 mask = 0x80;
    
    for (j=0; j<8; j++, mask >>= 1) {
      msb = (result & 0x80000000);

      result <<= 1;
      
      if ((i < nbytes) && (data[i] & mask))
	result |= 0x00000001;
  
      if (msb) 
	result ^= CRC_GENERATOR;
    }
  }
  return result;
}

/* CRC-32 as per ITU rec. I363.5 */
u32 atm_crc(const u8 data[],int nbytes)
{
  u32 crc1 = crc32_1(nbytes*8);
  u32 crc2 = crc32_2(data,nbytes);

  u32 result = ~(crc1 ^ crc2);

  return result;
}

/* ATM cell header error check
   as per ITU rec. I432 */
u8 atm_hec(const u8 data[],int nbytes)
{
  int i,j,msb=0;
  u8 result = 0;

  assert(nbytes == 4);
 
  for (i=0; i < nbytes + 1; i++) {
    u8 mask = 0x80;
    
    for (j=0; j<8; j++, mask >>= 1) {
      msb = (result & 0x80);

      result <<= 1;
      
      if ((i < nbytes) && (data[i] & mask))
	result |= 0x01;
  
      if (msb) 
	result ^= HEC_GENERATOR;

    }
  }

  return result ^ 0x55;
}


int  atm_cell_header_init(u8 vpi, u16 vci, 
			  u8 cell_header[], int header_size)
{
  if (header_size == 5) {
    u8 vpi_hi_nibble = vpi >> 4;
    u8 vpi_lo_nibble = vpi & 0x0F;
    u8 gfc = 0;
    u8 vci_hi_nibble =   (vci & 0xF000) >> 12;
    u8 vci_lo_nibble =   (vci & 0x000F);
    u8 vci_middle_byte = (vci & 0x0FF0) >> 4;
    
    cell_header[0] = MAKE_BYTE(gfc,vpi_hi_nibble);
    cell_header[1] = MAKE_BYTE(vpi_lo_nibble,vci_hi_nibble);
    cell_header[2] = vci_middle_byte;
    cell_header[3] = MAKE_BYTE(vci_lo_nibble,0);
    cell_header[4] = 0;

    return 0;
  }

  return -1;
}



#ifndef NO_MAIN

int read_data(unsigned char buffer[],int size)
{
  int num_bytes;

  for (num_bytes = 0; num_bytes < size;) {
    int rc;
    unsigned int val;
    int ch;
    
    rc = scanf("%x",&val);
    
    if (rc < 0) {
      break;
    }
    
    buffer[num_bytes++] = (unsigned char)val;
    
    do {
      ch = getchar();
    } while (ch != EOF && isspace(ch) && ch != '\n');
    
    if (ch == '\n')
      break;
    
    if (isxdigit(ch)) {
      ungetc(ch,stdin);
      continue;
    }
  }

  return num_bytes;
}


void interactive(void)
{
  char buffer[4096];

  for(;;) {
    unsigned char buffer[4096];
    int  nbytes;
    u32 crc;
    u8  cell_hec;
    int cmd;

    printf("Interactive>");
    fflush(stdout);
    
   

    cmd = getchar();

    switch (cmd) {
    case 'c':
    case 'C':
      nbytes = read_data(buffer,sizeof(buffer));   
      crc = crc32(buffer,nbytes);
      printf("CRC-32: 0x%08X\n",crc);
      break;

    case 'h':
    case 'H':
      nbytes = read_data(buffer,sizeof(buffer));   
      cell_hec = hec(buffer,nbytes);
      printf("CELL-HEC: 0x%02X\n",cell_hec);
      break;

    case EOF:
      return;

    default:
      if (!isspace(cmd)) {
	printf("Illegal character: '%c'\n",cmd);
	while (((cmd = getchar()) != EOF) && (cmd != '\n')) 
	  ;
      }

      break;
    }
  }
}



int main()
{
  u8 buffer[44];
  u32 crc;
  int i;

  memset(buffer,0,sizeof(buffer));

  buffer[43] = 40;

  crc = crc32(buffer,sizeof(buffer));

  printf("CRC (all zeroes) = 0x%08X\n",crc);


  memset(buffer,0,sizeof(buffer));
  memset(buffer,0xFF,40);
  buffer[43] = 40;

  crc = crc32(buffer,sizeof(buffer));

  printf("CRC (all ones) = 0x%08X\n",crc);

  for (i=0; i<40; i++) 
    buffer[i] = i+1;

  crc = crc32(buffer,sizeof(buffer));
  
  printf("CRC (1-40) = 0x%08X\n",crc);


  interactive();

  return 0;
}

#endif
  
