/**********************************************************************
                                                                     
   Copyright (C) 2002,2003 Emil D. Kohn   emild@cs.technion.ac.il     
                                                                     
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 **********************************************************************/



#include <errno.h>
#include <unistd.h>
#include <assert.h>
#include "pppd_io.h"

#define PPP_FRAME_MARKER 0x7E
#define PPP_ESCAPE       0x7D
#define MAX_RETRIES      10

static int read_n_bytes(int fd,
			void* buffer,
			size_t size)
{
  int rc;
  int retry_count = 0;

 repeat:
  rc = read(fd,buffer,size);


  if ((rc < 0) &&
      (retry_count++ < MAX_RETRIES) && 
      (errno == EINTR || errno == EAGAIN))
    goto repeat;

  return rc;
}




static int write_n_bytes(int fd,
			 const void* buffer,
			 size_t size)
{
  int rc;
  int retry_count = 0;

 repeat:
  rc = write(fd,buffer,size);

  if ((rc < 0) &&
      (retry_count++ < MAX_RETRIES) && 
      (errno == EINTR || errno == EAGAIN))
    goto repeat;

  return rc;

}


int read_from_pppd(int fd,
		   u8 buffer[],
		   int buff_size)
{
  int nbytes = 0;
  u8 ch;
  int escape = 0;

  for(;;) {
    int rc = read_n_bytes(fd,&ch,1);
    if (rc <= 0)
      return rc;

    if (ch == PPP_FRAME_MARKER && !escape) {
      if (nbytes > 0)
	break;
      continue;
    } else if (ch == PPP_ESCAPE) {
      escape = 1;
      continue;
    } else if (escape) {
      ch ^= 0x20;
      escape = 0;
    } 

    buffer[nbytes++] = ch;

    if (nbytes > buff_size)
      return -ERROR_BUFFER_OVERFLOW;
  }

  return nbytes;
}



int write_to_pppd(int fd,
		  const u8 buffer[],
		  int buff_size,
		  u32 asyncmap)
{
  int i;
  u8 ch = PPP_FRAME_MARKER;
  int rc;
  int nbytes = 0;

  rc = write_n_bytes(fd,&ch,1);

  if (rc <= 0)
    return rc;
  
  nbytes++;

  for (i=0; i<buff_size; i++) {
    ch = buffer[i];

    if (ch == PPP_FRAME_MARKER ||
	ch == PPP_ESCAPE ||
	(ch < 32 && (((u32)0x1 << ch) & asyncmap))) {
      u8 esc = PPP_ESCAPE;
      rc = write_n_bytes(fd,&esc,1);
      if (rc <= 0)
	return rc;
      
      nbytes++;
      ch ^= 0x20;
    }

    rc = write_n_bytes(fd,&ch,1);

    if (rc <= 0)
      return rc;
    nbytes++;
  }

  ch = PPP_FRAME_MARKER;
  rc = write_n_bytes(fd,&ch,1);
  if (rc <= 0)
      return rc;

  nbytes++;
  return nbytes;
}
