/**********************************************************************
                                                                     
   Copyright (C) 2002,2003 Emil D. Kohn   emild@cs.technion.ac.il     
                                                                     
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 **********************************************************************/

#include <stdio.h>
#include <assert.h>
#include "sar.h"
#include "atm_crc.h"

#define CELL_HEADER_SIZE    5
#define PAYLOAD_TYPE_OFFSET 3
#define CELL_PAYLOAD_SIZE   48
#define CELL_TOTAL_SIZE     (CELL_HEADER_SIZE + CELL_PAYLOAD_SIZE)
#define TX_CELL_SIZE        64
#define RX_CELL_SIZE        56
#define TRAILER_SIZE        (2*sizeof(u8) + sizeof(u16) + sizeof(u32)) /* two zero bytes, length, and CRC */
#define NO_MORE_DATA        2  /* also known as AAU in ITU recommendation speak */

/* RFC 1661 defines these constants */
#define PPP_ADDRESS_BYTE    0xFF
#define PPP_CONTROL_BYTE    0x03

static u8 cell_header_not_last[CELL_HEADER_SIZE];
static u8 cell_header_last[CELL_HEADER_SIZE];


int sar_init(u8 vpi, u16 vci)
{
  int rc;

  rc = atm_cell_header_init(vpi,vci,cell_header_last,CELL_HEADER_SIZE);

  if (rc < 0) {
    goto err_init_header_last;
  }

  cell_header_last[3] |= NO_MORE_DATA;
  
  rc = atm_cell_header_init(vpi,vci,cell_header_not_last,CELL_HEADER_SIZE);

  if (rc < 0) {
    goto err_init_header_not_last;
  }

  cell_header_last[CELL_HEADER_SIZE-1] = atm_hec(cell_header_last,CELL_HEADER_SIZE-1);
  cell_header_not_last[CELL_HEADER_SIZE-1] = atm_hec(cell_header_not_last,CELL_HEADER_SIZE-1);

  return 0;

 err_init_header_not_last:
 err_init_header_last:
  return rc;
}




int segmentation(u8 ppp_frame[], int frame_size, int frame_buffer_size,
		 u8 cell_buffer[], int cell_buffer_size)
{
  int n_cells;
  int cell_buff_limit;
  u8* trailer_ptr;
  u8* cell_ptr;
  u8* frame_ptr;
  int i;
  u32 crc;
  u8* ppp_frame_limit;
  u8* ppp_frame_buffer_end = &ppp_frame[frame_buffer_size];
  int last_cell_bytes;

  assert(frame_size >= 4);
  assert(frame_buffer_size < 0xFFFF);
  assert(frame_size < frame_buffer_size - CELL_PAYLOAD_SIZE);

  /* remove trailing CRC */
  ppp_frame[frame_size - 1] = '\0';
  ppp_frame[frame_size - 2] = '\0';
  frame_size -= 2;
  

  if (ppp_frame[0] == PPP_ADDRESS_BYTE) {
    assert(ppp_frame[1] == PPP_CONTROL_BYTE);
    /* Skip unused address and control bytes */
    ppp_frame += 2;
    frame_size -= 2;
  }

  trailer_ptr = ppp_frame + frame_size;

  n_cells = (frame_size + CELL_PAYLOAD_SIZE - 1)/CELL_PAYLOAD_SIZE;

  assert(n_cells >= 1);
  assert((n_cells * CELL_PAYLOAD_SIZE - frame_size) >= 0);
  assert((n_cells * CELL_PAYLOAD_SIZE - frame_size) < CELL_PAYLOAD_SIZE);

  if ((n_cells * CELL_PAYLOAD_SIZE - frame_size) < TRAILER_SIZE)
    n_cells++;

  cell_buff_limit = n_cells * TX_CELL_SIZE;
  assert(cell_buff_limit <= cell_buffer_size);

  ppp_frame_limit = ppp_frame + n_cells * CELL_PAYLOAD_SIZE;
  assert(ppp_frame_limit < ppp_frame_buffer_end);


  for (i=0, frame_ptr=ppp_frame, cell_ptr = cell_buffer; 
       i<n_cells-1; 
       i++, frame_ptr+=CELL_PAYLOAD_SIZE, cell_ptr+=TX_CELL_SIZE) {
    memcpy(cell_ptr, cell_header_not_last, CELL_HEADER_SIZE);
    memcpy(cell_ptr+CELL_HEADER_SIZE, frame_ptr, CELL_PAYLOAD_SIZE);
  }

  /* zero out the pad bytes */
  memset(cell_ptr, 0, TX_CELL_SIZE);

  /* Copy the last cell */
  memcpy(cell_ptr, cell_header_last, CELL_HEADER_SIZE);
  cell_ptr += CELL_HEADER_SIZE;

  assert(cell_ptr >= cell_buffer + CELL_HEADER_SIZE);
  assert(cell_ptr <= (&cell_buffer[cell_buff_limit] - (TX_CELL_SIZE - CELL_HEADER_SIZE)));

  last_cell_bytes = (ppp_frame + frame_size - frame_ptr);
  if (last_cell_bytes > 0) {
    assert(last_cell_bytes  <= CELL_PAYLOAD_SIZE - TRAILER_SIZE);
    memcpy(cell_ptr, frame_ptr, last_cell_bytes);
  }

  /* Set up the length field (big endian) */
  cell_ptr[CELL_PAYLOAD_SIZE - sizeof(u32) - 2] = (frame_size >> 8);
  cell_ptr[CELL_PAYLOAD_SIZE - sizeof(u32) - 1] = (frame_size & 0xFF);

  assert(ppp_frame_limit-trailer_ptr >= 0);
 
  memset(trailer_ptr,0,ppp_frame_limit-trailer_ptr);
  ppp_frame_limit[-(int)sizeof(u32) - 2] = (frame_size >> 8);
  ppp_frame_limit[-(int)sizeof(u32) - 1] = (frame_size & 0xFF);



  crc = atm_crc(ppp_frame,ppp_frame_limit - ppp_frame - sizeof(u32));
  
  /* Add the checksum (big endian) */
  cell_ptr[CELL_PAYLOAD_SIZE - 4] = (crc & 0xFF000000) >> 24;
  cell_ptr[CELL_PAYLOAD_SIZE - 3] = (crc & 0x00FF0000) >> 16;
  cell_ptr[CELL_PAYLOAD_SIZE - 2] = (crc & 0x0000FF00) >> 8;
  cell_ptr[CELL_PAYLOAD_SIZE - 1] = (crc & 0x000000FF);
  

  return cell_buff_limit;
}


INLINE static int buffer_is_empty(const u8* head, const u8* tail)
{
  return head == tail;
}


INLINE static int available_bytes(const u8* head, const u8* tail)
{
  return tail - head;
}


INLINE static void advance_pointer(u8 buffer[], int buffer_size,
				   u8** ptr, int offset)
{
  int new_index;
  
  assert(ptr != NULL);
  assert(*ptr != NULL);
  assert(*ptr >= buffer);
  assert(*ptr < &buffer[buffer_size]);
  assert((offset % RX_CELL_SIZE) == 0);

  *ptr += offset;
}





int reassembly(u8 cell_buffer[], int cell_buffer_size,
	       u8** head, u8** tail,
	       u8 ppp_frame[], int ppp_frame_buffer_size,
	       reader_function callback)
{
  u8* frame_ptr = ppp_frame;
  int frame_len = 0;
  u8* ppp_frame_end = &ppp_frame[ppp_frame_buffer_size];


  assert(cell_buffer != NULL);
  assert(cell_buffer_size > (2*RX_CELL_SIZE));
  assert(head != NULL);
  assert(*head != NULL);
  assert(tail != NULL);
  assert(*tail != NULL);
  assert(*head >= cell_buffer);
  assert(*tail >= cell_buffer);
  assert(*tail <= cell_buffer + cell_buffer_size);
  assert((*head < cell_buffer + cell_buffer_size) || 
	 *head == *tail);
  assert(*head <= *tail);
  assert(ppp_frame != NULL);
#ifdef USE_ADDRESS_AND_CONTROL
  assert(ppp_frame_buffer_size >=4);
#else
  assert(ppp_frame_buffer_size >= 2);
#endif
  assert(callback != NULL);


#ifdef USE_ADDRESS_AND_CONTROL
  frame_ptr[0] = PPP_ADDRESS_BYTE;
  frame_ptr[1] = PPP_CONTROL_BYTE;
  frame_ptr += 2;
  frame_len += 2;
#endif


  for (;;) {
    int bytes_in_buffer = available_bytes(*head,*tail);

    assert(bytes_in_buffer >= 0);

    if (bytes_in_buffer < RX_CELL_SIZE) {
      u8* new_head = cell_buffer + (RX_CELL_SIZE - bytes_in_buffer);
      int read_bytes;

      memcpy(new_head,*head,bytes_in_buffer);

    again:
      read_bytes = callback(cell_buffer + RX_CELL_SIZE, cell_buffer_size - RX_CELL_SIZE);

      if (read_bytes == 0)
	goto again;

      assert(read_bytes >= 0);
      if (read_bytes < 0)
	return read_bytes;
      
      *head = new_head;
      *tail = cell_buffer + RX_CELL_SIZE + read_bytes;

    }

    

    assert(frame_ptr + CELL_PAYLOAD_SIZE <= ppp_frame_end); 
    
    memcpy(frame_ptr,&(*head)[CELL_HEADER_SIZE],CELL_PAYLOAD_SIZE);
    if ((*head)[PAYLOAD_TYPE_OFFSET] & NO_MORE_DATA) {
      u16 frame_crc;
      int len;

      len= ((*head)[CELL_TOTAL_SIZE - sizeof(u32) - 2] << 8) | 
	(*head)[CELL_TOTAL_SIZE - sizeof(u32) - 1];
      
      frame_len += len;
      
      frame_crc = ppp_crc16(ppp_frame,frame_len);
      
      ppp_frame[frame_len] = frame_crc & 0xFF;
      ppp_frame[frame_len + 1] = frame_crc >> 8;
      
      frame_len += 2;
      

      advance_pointer(cell_buffer,*tail - cell_buffer,head,RX_CELL_SIZE);

      return frame_len;
    }
    
    frame_ptr += CELL_PAYLOAD_SIZE;
    advance_pointer(cell_buffer,*tail - cell_buffer,head,RX_CELL_SIZE);
  }
}


