\documentclass[ucs,handout]{beamer}
\usepackage[utf8x]{inputenc}

%include lhs2TeX.fmt
%include polycode.fmt
%options ghci -fglasgow-exts

%subst keyword a = "\textcolor{purple}{\mathbf{"a"}}"
%subst string a  = "\text{\textcolor{orange}{\tt \char34 " a "\char34}}"
%subst numeral a        = "\mathrm{\textcolor{orange}{" a "}}"

%format \ = "\textcolor{purple}{\lambda}"
%format not = "\textcolor{purple}{\mathbf{not}}"
%format otherwise = "\textcolor{purple}{\mathbf{otherwise}}"

%format forall a = "\forall" a
%format . = "\mathbin{.}"

%format ^ = "^"
%format || = "\mathrel{\mid\mid}"
%format && = "\mathrel{\&\&}"

%format >< = "\mathrel{\times}"

%format = = "\mathrel{\textcolor{blue}{=}}"

%if False

%format | = "\mathrel{\textcolor{blue}{\mid }}"
%format ( = "\textcolor{blue}("
%format : = "\textcolor{blue}{:}"
%format ) = "\textcolor{blue})"
%format [ = "\textcolor{blue}["
%format ] = "\textcolor{blue}]"
%format $ = "\mathrel{\textcolor{blue}\$}"
%format :: = "\textcolor{blue}{::}"

%endif

%format Monad = "\textcolor{brown}{\mathbf{Monad}}"
%format return = "\textcolor{brown}{\mathbf{return}}"
%format fail = "\textcolor{brown}{\mathbf{fail}}"
%format >>= = "\textcolor{brown}{\mathbin{\bind}}"
%format >> = "\textcolor{brown}{\mathbin{\sequ}}"

%format IO = "\textcolor{brown}{\mathbf{IO}}"
%format putStr = "\textcolor{brown}{\mathbf{putStr}}"
%format getLine = "\textcolor{brown}{\mathbf{getLine}}"
%format putStrLn = "\textcolor{brown}{\mathbf{putStrLn}}"

%format Maybe = "\textcolor{brown}{\mathbf{Maybe}}"
%format Just = "\textcolor{brown}{\mathbf{Just}}"
%format Nothing = "\textcolor{brown}{\mathbf{Nothing}}"

%format map = "\textcolor{brown}{\mathbf{map}}"
%format member = "\textcolor{brown}{\mathbf{member}}"


%format factorial1
%format factorial2
%format my_gcd1
%format my_gcd2
%format reverse1
%format reverse2

\long\def\ignore#1{}

%if False
\usepackage{listings}
\lstnewenvironment{code} 
    {\lstset{}
      \csname lst@SetFirstLabel\endcsname}
    {\csname lst@SaveFirstLabel\endcsname}
%    \lstset{
%      basicstyle=\small\ttfamily,
%      flexiblecolumns=false,
%      basewidth={0.5em,0.45em},
%      literate={+}{{$+$}}1 {/}{{$/$}}1 {*}{{$*$}}1 {=}{{$=$}}1
%               {>}{{$>$}}1 {<}{{$<$}}1 {\\}{{$\lambda$}}1
%               {->}{{$\rightarrow$}}2 {>=}{{$\geq$}}2 {<-}{{$\leftarrow$}}2
%               {<=}{{$\leq$}}2 {=>}{{$\Rightarrow$}}2 
%               {\ .}{{$\circ$}}2 {\ .\ }{{$\circ$}}2
%               {>>}{{>>}}2 {>>=}{{>>=}}2
%               {|}{{$\mid$}}1,
%      keywordstyle=\color{purple}
%    }
%endif

\mode<presentation>
{
  \usetheme{Warsaw}
  % or ...

  \setbeamercovered{transparent}
  % or whatever (possibly just delete it)
}

\usepackage[english]{babel}
\title{An overview of Haskell}
\date{23/7/2007}
\author{Haggai Eran}
\titlegraphic{\includegraphics[width=3cm]{logo.jpg}}

\ignore{
 Some code for working in Haskell, that isn't needed in the presentation.

\begin{code}
import Test.QuickCheck
import List
import Text.ParserCombinators.Parsec
import Graphics.UI.Gtk
import Char

qc_config = defaultConfig
    { configEvery = \n args -> "" }
\end{code}

} % end ignore


\begin{document}

\begin{frame}
    \titlepage
\end{frame}

\begin{frame}{Outline}
  \tableofcontents
  % You might wish to add the option [pausesections]
\end{frame}

\section{Introduction}

\begin{frame}\frametitle{Introduction}
    Haskell is a \alert{pure functional language}. It means that:
    \begin{itemize}[<+->]
        \item Variables never change after definition.
        \item Functions don't have side effects.
        \item Functions always return the same output given the same input.
    \end{itemize}
\end{frame}

\begin{frame}{History}
  \begin{itemize}[<+->]
   \item Designed by a committee. [1990s]
   \item (Nevertheless, it is an elegant language.)
   \item Haskell 98 - (Informal) standardization, and basis for further development.
  \end{itemize}
\visible<4->{
Named after Haskell B. Curry:
\begin{figure}
 \includegraphics[width=3cm]{haskell-curry.jpg}
\end{figure}
}
\end{frame}

\begin{frame}{History}
  \framesubtitle{Designed by a committee}
  \begin{center}
 \includegraphics[width=10cm]{comittee.png}
 % comittee.png: 745x495 pixel, 72dpi, 26.28x17.46 cm, bb=
\end{center}

\end{frame}


\subsection{Nice Syntactic Features}

\begin{frame}{Nice syntactic features}
\framesubtitle{Guards}
    \begin{itemize}[<+->]
        \item Standard if-then-else:
\begin{code}
my_gcd1 m n =  if n == 0      then m
               else if m < n  then my_gcd1 n m
               else my_gcd1 n (m `mod` n)
\end{code}
        \item Guards:
\begin{code}
my_gcd2 m 0               = m
my_gcd2 m n  | m < n      = my_gcd2 n m
             | otherwise  = my_gcd2 n (m `mod` n)
\end{code}
\end{itemize}
\end{frame}

\begin{frame}{Nice syntactic features}
\framesubtitle{Pattern Matching}
  \begin{itemize}[<+->]
        \item Simple Case expressions:
\begin{code}
factorial1 n = case n of
  0 -> 1
  n -> n * factorial1 (n-1)
\end{code}

        \item Pattern Matching:
\begin{code}
factorial2 0 = 1
factorial2 n = n * factorial2 (n-1)
\end{code}

    \end{itemize}
\end{frame}

\begin{frame}{Lists}
A list in Haskell is defined recursively.
\begin{definition} \begin{code}
data [a] = [] | a:[a]
\end{code}
\end{definition}
And there's some syntactic sugar for using lists: \begin{code}
    [1..3] == [1,2,3] == 1:[2,3] == 1:2:3:[]
\end{code}
\end{frame}
\begin{frame}{Lazy Lists}
Since Haskell is a lazy language, you can define infinite lists:
\pause
\begin{code}
primes =  sieve [2..] where 
    sieve (p:tail) = let 
        filtered_tail = sieve [ n | n <- tail, n `mod` p > 0]
        in p : filtered_tail
\end{code}
\pause
\begin{code}
factorial_list = 1:[a*n  | a <- factorial_list
                         | n <- [1..]]
\end{code}

\end{frame}

\begin{frame}{QuickSort}
\begin{code}

quicksort [] = []
quicksort (hd:tail) = quicksort small ++ [hd] ++ quicksort large
  where
    small  = [x | x <- tail, x <= hd]
    large  = [x | x <- tail, x > hd]
\end{code}
\end{frame}

% currying and pointfree
\begin{frame}{Currying} 
\only<1>{\begin{code}
inc x = 1 + x
\end{code}}
\only<2>{\begin{code}
inc x = (+) 1 x
\end{code}}
\only<3>{ \begin{code}
inc = (+) 1
\end{code}}
\only<4>{
\begin{code}
inc = (+1)
\end{code}
}
\end{frame}

\begin{frame}{Pointfree programming} 
\only<1>{\begin{code}
h x = f(g(x))
\end{code}}
\only<2>{\begin{code}
h x = (f . g) (x)
\end{code}}
\only<3>{\begin{code}
h = f . g
\end{code}}
\end{frame}

\section{Features}
\subsection{Type System}

\begin{frame}[label=reverse]{Type System Introduction}
    Haskell uses static typing, but is very expressive because of its 
    polymorphism and type classes.
    \pause
    \begin{example}
\begin{code}
reverse1 :: [a] -> [a]
reverse1 []         = []
reverse1 (hd:tail)  = reverse1 tail ++ [hd]
\end{code}
\pause
\end{example}
    Since {\it reverse\_list} is polymorphic, you can use it for any type 
    of list: 
\begin{itemize}
  \item |reverse1  [1,2,3]          -> | \eval{reverse1  [1,2,3]}
  \item |reverse1  "Hello, World"   -> | \eval{reverse1 "Hello, World"}
\end{itemize}

\hfill \hyperlink{efficient-reverse}{\beamergotobutton{An efficient reverse}}

\end{frame}

\begin{frame}{Algebraic Data Types}
Haskell supports user defined algebraic data types, which combined with pattern matching are
very expressive.\begin{code}
data Maybe a = Nothing | Just a
\end{code}

\pause

\begin{example}
\begin{code}
divide :: (Integral a) => a -> a -> Maybe a
divide x 0 = Nothing
divide x y = Just (x `div` y)
\end{code}
\end{example}

\end{frame}

\begin{frame}{Algebraic Data Types}
 \framesubtitle{Decomposition using pattern matching}
 
 \begin{example}
    \begin{code}
default_value :: Maybe a -> a -> a
default_value  Nothing    x    = x
default_value  (Just x)   _  = x

\end{code}

 \end{example}

\end{frame}

\begin{frame}{Algebraic Data Types}
  \framesubtitle{Describing complex data structures}
Complex data structures can be described (without pointers, of course).

\begin{code}
data Tree a = Leaf a | Branch (Tree a) (Tree a)
\end{code}
\pause
\begin{code}
size :: Tree a -> Int
size (Leaf _)             = 1
size (Branch left right)  = 1 + size left + size right
\end{code}
\end{frame}

\begin{frame}{Encapsulation}
  There is no abstract type construct in Haskell, but instead there is a hierarchial module system,
  which can be used for encapsulation.
  \pause
  \begin{example}\begin{code}
module Stack(  Stack, push, pop, empty, 
               top, is_empty ) where
  data Stack a      = Stk [a]
  empty             = Stk []
  
  push (Stk s) x    = Stk (x:s)
  pop (Stk (x:s))   = Stk s
  top (Stk (x:s))   = x
  is_empty (Stk s)  = null s
  
\end{code}\end{example}
\end{frame}

\begin{frame}{Type Classes}
In Haskell, Type classes allow both overloading names, and writing generic functions which are made specific for some class.\pause
\begin{example}\begin{code}
class Eq a where
 (==) :: a -> a -> Bool
 (/=) :: a -> a -> Bool
\end{code}\pause\begin{code}
instance Eq Int where
  i1 == i2  = eqInt i1 i2
  i1 /= i2  = not (i1 == i2)
\end{code}
\end{example}
\end{frame}

\begin{frame}{Type Classes}
\framesubtitle{Generic Classes and Functions}

\begin{example}\begin{code}
instance (Eq a) => Eq [a] where
  []        ==  []      = True
  (x:xs)    ==  (y:ys)  = x == y && xs == ys
  xs /= ys              = not (xs == ys)

\end{code}\pause\begin{code}
member :: Eq a => a -> [a]     ->  Bool
member x  []                   =   False
member x  (y:ys)  | x == y     =   True
                  | otherwise  =   member x ys
\end{code}
\end{example}
\end{frame}

\subsection{Higher Order Functions}
\begin{frame}{Higher Order Functions}
Functions are first-class values, and can be passed to other functions.

\begin{example} \begin{code}
map :: (a -> b) -> [a] -> [b]
map f  []           = []
map f  (head:tail)  = (f head):(map f tail)
\end{code}\pause
\end{example} \begin{code}
inc      :: (Num a) => a -> a
(*3)     :: (Num a) => a -> a
\end{code}

\begin{quote}
|map  inc [1,2,3]  == | \eval{map inc [1,2,3]} % [2,3,4]

|map  (*3) [1,2,3]  == | \eval{map (*3) [1,2,3]} % [3,6,9]
\end{quote}
\end{frame}

\begin{frame}{map - More Uses} 
\begin{quote}
\eval{:t toUpper} % \begin{code} toUpper  :: Char -> Char \end{code}

|map  toUpper  "Hello"  == | \eval{map toUpper "Hello"} % "HELLO"
\end{quote}
\pause
You can even define:
\begin{code}
stringToUpper :: String -> String 
stringToUpper = map toUpper
\end{code}
\end{frame}

% IO
\subsection{IO and Monads}
\begin{frame}{IO and Monads}
Pure functional language $\Rightarrow$ No side-effects in functions.

So how can we perform IO?

\pause
With the IO Monad! 

\pause
A value of the type |IO a| represent an action, which
returns a value of type |a|, once performed. 
\pause
\begin{example} \begin{code}
getLine  :: IO String
putStr   :: String -> IO ()
\end{code}
\end{example}
\end{frame}

\begin{frame}{IO Syntax}

\begin{example}
\begin{code}
greet :: String -> String
greet name = "Hello, " ++ name

main :: IO ()
main = do
    name <- getLine
    putStrLn (greet name) 
\end{code}
\end{example}
\end{frame}

\begin{frame}[label=monadic-pointfree]{Monadic Pointfree Syntax}
  \begin{example}
    \begin{code}
echo :: IO ()
echo =  putStr "> "  >> 
        getLine      >>= 
        putStr       >> 
        putStr "\n"
\end{code}

\end{example}

\hfill \hyperlink{monad-class}{\beamergotobutton{The Monad Type Class}}

\end{frame}

\begin{frame}{The Maybe Monad}
\begin{block}{Maybe}\begin{code}
f :: Int -> Maybe Int
complex_function :: Maybe Int -> Maybe Int
complex_function mint = do
  i1 <- mint
  i2 <- f i1
  return i2
\end{code}

\end{block}
\end{frame}

\begin{frame}{The List Monad}
\begin{block}{List}
\begin{code}
(><) :: [a] -> [b] -> [(a,b)]
xs >< ys = do
  x <- xs
  y <- ys
  return (x,y)
\end{code}
\end{block}
\begin{example}
|[1,2] >< [3,4] -> | \eval{[1,2] >< [3,4]}
\end{example}
\end{frame}

\begin{frame}{Parsing}
\begin{block}{Parsec}
\begin{code}
perl_variable = do
    sigil  <- oneOf "&$@%"
    name   <- many alphaNum
    return (sigil : name)
\end{code}
\end{block}
\begin{example}
\begin{itemize}

  \item   |parse perl_variable "Parser" "$var" -> | \eval{parse perl_variable "Parser" "$var"}
  \item   |parse perl_variable "Parser" "not a var" -> | \eval{parse perl_variable "Parser" "not a var"}
\end{itemize}
\end{example}

\end{frame}

\begin{frame}{GUI - Gtk2Hs}
\begin{columns}
\begin{column}{8cm}
\begin{code}
main_gui :: IO ()
main_gui = do
  initGUI
  window  <- windowNew
  button  <- buttonNew
  set window  [  containerBorderWidth := 10,
                 containerChild := button ]
  set button  [  buttonLabel := "Hello World" ]
  onClicked button (putStrLn "Hello World")
  onDestroy window mainQuit
  widgetShowAll window
  mainGUI
\end{code}
\end{column}\begin{column}{2cm}
\includegraphics[width=20mm]{gui.png}
\end{column}\end{columns}
\end{frame}

% QuickCheck
\subsection{Testing}

\begin{frame}{Testing with QuickCheck}

\begin{code}
property_factorial1 n = 
    factorial1 (n+1) `div` factorial1 n == n+1
\end{code}
\pause
|quickCheck property_factorial1| 

results in 
\begin{quote}{\tt \perform{check qc_config property_factorial1}}
\end{quote}
\pause
\begin{code}
property_factorial2 n = n >= 0 ==> 
    factorial1 (n+1) `div` factorial1 n == n+1
\end{code}
\pause
|quickCheck property_factorial2| 

results in
\begin{quote}{\tt \perform{check qc_config property_factorial2}}
\end{quote}

\end{frame}

\begin{frame}{Some more QuickCheck examples}
\begin{code}
property_gcd n = n >= 0 ==> (n `mod` (my_gcd2 n (n+2))) == 0
\end{code}

Checking only specific values:
\begin{code}
property_primes = forAll (two some_primes) $ \(p,q) -> 
    (p == q || gcd p q == 1)
  where some_primes  = elements $ take 200 primes
\end{code}
\pause
Lists can be generated too:
\begin{code}
property_reverse list = (reverse1 . reverse1) list == list
property_quicksort list = quicksort list == List.sort list
\end{code}

\end{frame}

\begin{frame}{What else?}
 \begin{itemize}[<+->]
  \item Implementations: GHC, Hugs, Helium, JHC, YHC
  \item Parallel GHC, Concurrent GHC, STM
  \item Cabal
  \item Visual Haskell, EclipseFP
  \item Famous Projects Using Haskell: Pugs, Darcs.
  \item DSLs, DSELs.
  \item Literate Haskell
 \end{itemize}

\end{frame}

\section{Haskell Implementation}
\begin{frame}{Few Implementation Notes}
\begin{itemize}
    \item These notes are based on the article about the ``Spineless Tagless G-Machine'' by Simon Peyton
      Jones, which is the basis for current implementations of the Glasgow Haskell Compiler 
      - GHC.
    \item I'll only speak about some of the basic details, because I have much more to learn ...
\end{itemize}
\end{frame}

\begin{frame}{The Compiler Structure}
\begin{enumerate}[<+->]
    \item Preprocessing - Removing the literate markup, if needed, and also
        running a C preprocessor, if asked by the user.
    \item Compiling into the smaller {\it Core} language, an intermediate 
        language without the syntactic sugar. Type checking is performed, and
        pattern matching is translated into simple |case| expressions.
    \item Some optimizations are performed on the intermediate language.
    \item The Core language is translated into the STG language.
    \item The STG language is translated by a code generator into C, or 
        into machine code.
\end{enumerate}
\uncover<5->{
We'll focus on the STG language, and how it is translated into C.
}
\end{frame}

\subsection{The Spineless Tagless G-Machine Language}
\begin{frame}{The Spineless Tagless G-Machine Language}
The STG language is a very austere functional language, or a subset of Haskell.

\pause
It contains only the following constructs:
\begin{itemize}[<+->]
    \item Function applications, for using functions.
    \item |let| and |\| expressions, for creating new bindings.
    \item |case| expressions, for evaluating expressions.
    \item Constructor applications, for defining values.
\end{itemize}
\end{frame}

\begin{frame}{Translation into STG}
\begin{example}\begin{code}
    map f []             = []
    map f (head:tail)    = (f head) : (map f tail)
\end{code}
is translated to 
\begin{onlyenv}<2|| handout:0>\begin{code}
map = {} \n {head,list} ->
\end{code}\end{onlyenv}
\begin{onlyenv}<3|| handout:0>\begin{code}
map = {} \n {head,list} ->
    case list of
      Nil   {}            -> Nil {}
      Cons  {head,tail}   ->
\end{code} \end{onlyenv}
\begin{onlyenv}<4>\begin{code}
map = {} \n {head,list} ->
    case list of
      Nil   {}            -> Nil {}
      Cons  {head,tail}   ->
        let  f_head    = {f,head}  \u {}  -> f {y}
             map_tail  = {f,tail}  \u {}  -> map {f,tail}
        in Cons {f_head,map_tail}
\end{code} \end{onlyenv}
\end{example}
\end{frame}

\subsection{Memory Representation}

\begin{frame}{Memory Representation}
Many kinds of values:
\begin{itemize}[<+->]
\item Functions:      \hfill |{free_list}\n {arg_list} -> expr| \linebreak
  Contain code, and pointers to their free variables.
\item Thunks:         \hfill |{free_list}\u {} -> expr| \linebreak
  Unevaluated expressions, contain the code to evaluate, and any needed pointer.
\item Constructors:   \hfill |Constructor{arg_list}| \linebreak
  Contain the pointers to the constructors' parameters, which might be functions or thunks themselves.
\item Primitive Values: \linebreak
  Integers, characters, floating point numbers, etc.
\end{itemize}
\end{frame}

\begin{frame}{Closures}
In a polymorphic language, you cannot always know statically if a pointer is a function or a thunk, for example:
\begin{code}
compose f g x = f (g x)
\end{code}
|g x| might be a function or a thunk, on every call to compose.

\pause
It is convenient to hold all values (except the primitives) in memory in the same structure, as \alert{closures}:

\begin{visibleenv}<2->
\begin{figure}
\includegraphics[width=8cm]{closure.png}
\end{figure}
\end{visibleenv}
\end{frame}

\subsection{Running on Ordinary Machines}
\begin{frame}{A mapping to ordinary machines}
The STG language was defined with operational semantics. Each language construct
has an operational meaning:

\begin{table}
\begin{tabular}{||l||l||}
\hline
Construct               &   Operational meaning     \\
\hline
Function application    &   Tail call               \\
Let expression          &   Heap allocation         \\
Case expression         &   Evaluation              \\
Constructor application &   Return to continuation  \\
\hline
\end{tabular}
\end{table}
\end{frame}

\begin{frame}{The STG Abstract Machine}
The abstract machine which the implementation is based on has:
\begin{itemize}[<+->]
    \item Argument stack - a stack for passing parameters to functions.
    \item Return stack - a stack for continuations.
    \item Update stack - a stack for update frames (updating thunks).
\end{itemize}
\uncover<4->{
The machine also includes a heap (garbage collected) for holding closures.
}

\uncover<5->{
This is only the abstract machine, which is easier to understand. The real implementation has a different representation for these stacks.
}
\end{frame}

\begin{frame}{Function Application}
A function call is implemented by
\begin{itemize}[<+->]
    \item Pushing its arguments to the argument stack.
    \item Tail-calling the function (A jump into the function's code).
\end{itemize}
\begin{example}\begin{code}
  map {f,tail}
\end{code}\end{example}

\end{frame}

\begin{frame}{Let expressions}
|let| expressions give local names to closures, and evaluate an expression in the local environment. 

They are implemented by:
\begin{itemize}
    \item Constructing the closures in the heap.
    \item Evaluating the expression
\end{itemize}

\begin{example}\begin{code}
  let  f_head    = {f,head}  \u {}  -> f {y}
       map_tail  = {f,tail}  \u {}  -> map {f,tail}
  in Cons {f_head,map_tail}
\end{code}\end{example}

\end{frame}

\begin{frame}{Case expressions}
|case| expressions force evaluation of an expression, and then choose from
alternatives based on its value.

They are implemented by:
\begin{itemize}[<+->]
    \item Pushing a continuation (or continuations) onto the return stack.
    \item Evaluate the expression.
    \item The evaluation is responsible for continuing according to the right 
        alternative.
\end{itemize}

\begin{example}\begin{code}
    case list of
      Nil   {}            -> ...
      Cons  {head,tail}   -> ...
\end{code}
\end{example}
\end{frame}

\begin{frame}{Constructor Applications}
The application of a constructor is evaluated from within some case expression. The implementation:
\begin{itemize}[<+->]
    \item Pop the continuation from the return stack.
    \item Jump to the right alternative.
\end{itemize}

\uncover<3->{After return, either:}
\begin{itemize}[<+->]
  \item a special register points to the constructor's closure, for the inspecting its values, or 
  \item they could be returned in registers directly.
\end{itemize}

\begin{example}\begin{code}
    case list of
      Nil   {}            -> Nil {}
      Cons  {head,tail}   -> let ...
\end{code}
\end{example}

\end{frame}

\begin{frame}{Constructor Applications}
\framesubtitle{Notes}
\begin{itemize}[<+->]
    \item Returning in registers can avoid allocating a new closure in the 
        heap, and this is why the machine is called spineless.
    \item The fact that the alternatives can be chosen without holding a tag
        field for every different constructor is the reason why it is called 
        tagless.
        % TODO : what about G - Machine?
\end{itemize}
\end{frame}

\begin{frame}{Updating Thunks}
  In order to update thunks after they are evaluated:
  \begin{enumerate}[<+->]
   \item When entering an updatable closure 
      \begin{itemize}[<+->]
          \item An {\it update frame} is pushed to the update stack, which contain a pointer to the 
            closure to be updated, and the contents of the arguments and return stacks.
          \item The return stack and argument stack are made empty.
          \item Its sometimes nice to update the closure temporarily with a ``black hole'' closure.
      \end{itemize}
   \item When evaluation of a closure is complete an update is triggered.
      \begin{itemize}[<+->]
       \item If the closure is a function, it won't find enough arguments on the argument stack.
       \item If the closure is a value, it will attempt to pop a continuation from the return stack, 
          which is empty.
      \end{itemize}
   \item The update is either in-place, or by an indirection closure which is removed by GC.
  \end{enumerate}
\end{frame}

\section{Summary}

\begin{thebibliography}{1}
\begin{frame}{Links}

\bibitem{haskell.org}
\url{http://haskell.org}

\bibitem{tang}
Learning Haskell
\newblock Audrey Tang
\newblock \url{http://perlcabal.org/~autrijus/osdc/haskell.xul}

\bibitem{evolution}
The Evolution of a Haskell Programmer
\newblock  Fritz Ruehr
\newblock \url{http://www.willamette.edu/~fruehr/haskell/evolution.html}

\bibitem{history}
A history of haskell: being lazy with class.
\newblock \url{http://research.microsoft.com/~simonpj/papers/history-of-haskell/history.pdf}

\end{frame}

\begin{frame}{Links}
\framesubtitle{Implementation}

\bibitem{GHC}
GHC Commentary
\newblock \url{http://hackage.haskell.org/trac/ghc/wiki/Commentary}

\bibitem{STG}
Implementing lazy functional languages on stock hardware: The
  spineless tagless g-machine.
\newblock \url{http://citeseer.ist.psu.edu/peytonjones92implementing.html}.

\bibitem{GHC}
GHC Hackathon Videos
\newblock \url{http://video.google.com/videosearch?q=GHC+Hackathon&so=0}

\end{frame}
\end{thebibliography}

\begin{frame}{Thank you!}
 \begin{Huge} 
\begin{center}
  Thank you!
  
\medskip 

  Questions?
\end{center}  \end{Huge}
\end{frame}


\appendix

\begin{frame}{Appendix}
  \tableofcontents
\end{frame}


\section{Appendix}

\subsection{An Efficient Reverse}

\begin{frame}[label=efficient-reverse]{An Efficient Reverse}
 By the way: The previous slide's |reverse1| function has $O(n^2)$ complexity, since each |++| operation is linear in the first list's length.
\pause
 A more efficient version is:
 \begin{code}
reverse2 :: [a] -> [a]
reverse2 list = helper list []
  where
    helper  []         reversed   = reversed
    helper  (hd:tail)  reversed  = helper tail (hd:reversed)
\end{code}

which runs in $O(n)$ complexity.

\hfill \hyperlink{reverse<3>}{\beamergotobutton{Back}}

\end{frame}


\subsection{Monad Class}

\begin{frame}[label=monad-class]{Monad Class}\begin{code}
class Monad m where
  (>>=)   :: forall a b   . m a -> (a -> m b) -> m b
  (>>)    :: forall a b   . m a -> m b -> m b
  return  :: forall a     . a -> m a
  fail    :: forall a     . String -> m a
\end{code}

\hfill \hyperlink{monadic-pointfree}{\beamergotobutton{Back}}

\end{frame}

\end{document}
