/* ============================================================

Copyright (c) 2009 Advanced Micro Devices, Inc.  All rights reserved.

Redistribution and use of this material is permitted under the following
conditions:

Redistributions must retain the above copyright notice and all terms of this
license.

In no event shall anyone redistributing or accessing or using this material
commence or participate in any arbitration or legal action relating to this
material against Advanced Micro Devices, Inc. or any copyright holders or
contributors. The foregoing shall survive any expiration or termination of
this license or any agreement or access or use related to this material.

ANY BREACH OF ANY TERM OF THIS LICENSE SHALL RESULT IN THE IMMEDIATE REVOCATION
OF ALL RIGHTS TO REDISTRIBUTE, ACCESS OR USE THIS MATERIAL.

THIS MATERIAL IS PROVIDED BY ADVANCED MICRO DEVICES, INC. AND ANY COPYRIGHT
HOLDERS AND CONTRIBUTORS "AS IS" IN ITS CURRENT CONDITION AND WITHOUT ANY
REPRESENTATIONS, GUARANTEE, OR WARRANTY OF ANY KIND OR IN ANY WAY RELATED TO
SUPPORT, INDEMNITY, ERROR FREE OR UNINTERRUPTED OPERA TION, OR THAT IT IS FREE
FROM DEFECTS OR VIRUSES.  ALL OBLIGATIONS ARE HEREBY DISCLAIMED - WHETHER
EXPRESS, IMPLIED, OR STATUTORY - INCLUDING, BUT NOT LIMITED TO, ANY IMPLIED
WARRANTIES OF TITLE, MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
ACCURACY, COMPLETENESS, OPERABILITY, QUALITY OF SERVICE, OR NON-INFRINGEMENT.
IN NO EVENT SHALL ADVANCED MICRO DEVICES, INC. OR ANY COPYRIGHT HOLDERS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, PUNITIVE,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, REVENUE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED OR BASED ON ANY THEORY OF LIABILITY
ARISING IN ANY WAY RELATED TO THIS MATERIAL, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE. THE ENTIRE AND AGGREGATE LIABILITY OF ADVANCED MICRO DEVICES,
INC. AND ANY COPYRIGHT HOLDERS AND CONTRIBUTORS SHALL NOT EXCEED TEN DOLLARS
(US $10.00). ANYONE REDISTRIBUTING OR ACCESSING OR USING THIS MATERIAL ACCEPTS
THIS ALLOCATION OF RISK AND AGREES TO RELEASE ADVANCED MICRO DEVICES, INC. AND
ANY COPYRIGHT HOLDERS AND CONTRIBUTORS FROM ANY AND ALL LIABILITIES,
OBLIGATIONS, CLAIMS, OR DEMANDS IN EXCESS OF TEN DOLLARS (US $10.00). THE
FOREGOING ARE ESSENTIAL TERMS OF THIS LICENSE AND, IF ANY OF THESE TERMS ARE
CONSTRUED AS UNENFORCEABLE, FAIL IN ESSENTIAL PURPOSE, OR BECOME VOID OR
DETRIMENTAL TO ADVANCED MICRO DEVICES, INC. OR ANY COPYRIGHT HOLDERS OR
CONTRIBUTORS FOR ANY REASON, THEN ALL RIGHTS TO REDISTRIBUTE, ACCESS OR USE
THIS MATERIAL SHALL TERMINATE IMMEDIATELY. MOREOVER, THE FOREGOING SHALL
SURVIVE ANY EXPIRATION OR TERMINATION OF THIS LICENSE OR ANY AGREEMENT OR
ACCESS OR USE RELATED TO THIS MATERIAL.

NOTICE IS HEREBY PROVIDED, AND BY REDISTRIBUTING OR ACCESSING OR USING THIS
MATERIAL SUCH NOTICE IS ACKNOWLEDGED, THAT THIS MATERIAL MAY BE SUBJECT TO
RESTRICTIONS UNDER THE LAWS AND REGULATIONS OF THE UNITED STATES OR OTHER
COUNTRIES, WHICH INCLUDE BUT ARE NOT LIMITED TO, U.S. EXPORT CONTROL LAWS SUCH
AS THE EXPORT ADMINISTRATION REGULATIONS AND NATIONAL SECURITY CONTROLS AS
DEFINED THEREUNDER, AS WELL AS STATE DEPARTMENT CONTROLS UNDER THE U.S.
MUNITIONS LIST. THIS MATERIAL MAY NOT BE USED, RELEASED, TRANSFERRED, IMPORTED,
EXPORTED AND/OR RE-EXPORTED IN ANY MANNER PROHIBITED UNDER ANY APPLICABLE LAWS,
INCLUDING U.S. EXPORT CONTROL LAWS REGARDING SPECIFICALLY DESIGNATED PERSONS,
COUNTRIES AND NATIONALS OF COUNTRIES SUBJECT TO NATIONAL SECURITY CONTROLS.
MOREOVER, THE FOREGOING SHALL SURVIVE ANY EXPIRATION OR TERMINATION OF ANY
LICENSE OR AGREEMENT OR ACCESS OR USE RELATED TO THIS MATERIAL.

NOTICE REGARDING THE U.S. GOVERNMENT AND DOD AGENCIES: This material is
provided with "RESTRICTED RIGHTS" and/or "LIMITED RIGHTS" as applicable to
computer software and technical data, respectively. Use, duplication,
distribution or disclosure by the U.S. Government and/or DOD agencies is
subject to the full extent of restrictions in all applicable regulations,
including those found at FAR52.227 and DFARS252.227 et seq. and any successor
regulations thereof. Use of this material by the U.S. Government and/or DOD
agencies is acknowledgment of the proprietary rights of any copyright holders
and contributors, including those of Advanced Micro Devices, Inc., as well as
the provisions of FAR52.227-14 through 23 regarding privately developed and/or
commercial computer software.

This license forms the entire agreement regarding the subject matter hereof and
supersedes all proposals and prior discussions and writings between the parties
with respect thereto. This license does not affect any ownership, rights, title,
or interest in, or relating to, this material. No terms of this license can be
modified or waived, and no breach of this license can be excused, unless done
so in a writing signed by all affected parties. Each term of this license is
separately enforceable. If any term of this license is determined to be or
becomes unenforceable or illegal, such term shall be reformed to the minimum
extent necessary in order for this license to remain in effect in accordance
with its terms as modified by such reformation. This license shall be governed
by and construed in accordance with the laws of the State of Texas without
regard to rules on conflicts of law of any state or jurisdiction or the United
Nations Convention on the International Sale of Goods. All disputes arising out
of this license shall be subject to the jurisdiction of the federal and state
courts in Austin, Texas, and all defenses are hereby waived concerning personal
jurisdiction and venue of these courts.

============================================================ */


#include "NBody.h"

#include <stdio.h>
#include <string.h>
#include <iostream>
#include <fstream>
#include <cstdlib>

#include <GL/glut.h>
#include <cmath>
#include <malloc.h>

#include <CL/cl.h>

#include "clutils.h"
#include "displayutils.h"


//void* me;           /**< Pointing to NBody class */


int numBodies;      /**< No. of particles*/
cl_float* pos;      /**< Output position */
void * me;

cl_bool display;

float NBody::random(float randMax, float randMin)
{
    float result;
    result =(float)rand()/(float)RAND_MAX;

    return ((1.0f - result) * randMin + result *randMax);
}

int NBody::setupNBody()
{
    // make sure numParticles is multiple of group size

    numParticles = (cl_int)(((size_t)numParticles
        < groupSize) ? groupSize : numParticles);
    numParticles = (cl_int)((numParticles / groupSize) * groupSize);

    numBodies = numParticles;
    printf("No of bodies %d\n",numBodies);

    initPos = (cl_float*)malloc(numBodies * sizeof(cl_float4));
    if(initPos == NULL)
    {
        printf("Failed to allocate host memory. (initPos)");
        exit(1);
    }

    initVel = (cl_float*)malloc(numBodies * sizeof(cl_float4));
    if(initVel == NULL)
    {
        printf("Failed to allocate host memory. (initVel)");
        exit(1);
    }

#if defined (_WIN32)
    pos = (cl_float*)_aligned_malloc(numBodies * sizeof(cl_float4), 16);
#else
    pos = (cl_float*)memalign(16, numBodies * sizeof(cl_float4));
#endif
    if(pos == NULL)
    {
        printf("Failed to allocate host memory. (pos)");
        exit(1);
    }

#if defined (_WIN32)
    vel = (cl_float*)_aligned_malloc(numBodies * sizeof(cl_float4), 16);
#else
    vel = (cl_float*)memalign(16, numBodies * sizeof(cl_float4));
#endif

    if(vel == NULL)
    {
        printf("Failed to allocate host memory. (vel)");
        exit(1);
    }

    /* initialization of inputs */
    for(int i = 0; i < numBodies; ++i)
    {
        int index = 4 * i;

        // First 3 values are position in x,y and z direction
        for(int j = 0; j < 3; ++j)
        {
            initPos[index + j] = random(3, 50);
        }

        // Mass value
        initPos[index + 3] = random(1, 1000);

        // First 3 values are velocity in x,y and z direction
        for(int j = 0; j < 3; ++j)
        {
            initVel[index + j] = 0.0f;
        }

        // unused
        initVel[3] = 0.0f;
    }

    memcpy(pos, initPos, 4 * numBodies * sizeof(cl_float));
    memcpy(vel, initVel, 4 * numBodies * sizeof(cl_float));

    return SUCCESS;
}


int NBody::setupCL()
{
    cl_int status = CL_SUCCESS;


    /* Get Device specific Information */
    status = clGetDeviceInfo(cl_getDeviceId(),CL_DEVICE_MAX_WORK_GROUP_SIZE, sizeof(size_t),(void*)&maxWorkGroupSize,NULL);

    if(cl_errChk(status,"clGetDeviceInfo CL_DEVICE_MAX_WORK_GROUP_SIZE failed."))
        exit(1);


    status = clGetDeviceInfo(cl_getDeviceId(), CL_DEVICE_MAX_WORK_ITEM_DIMENSIONS, sizeof(cl_uint),(void*)&maxDimensions,NULL);

    if(cl_errChk(status,"clGetDeviceInfo CL_DEVICE_MAX_WORK_ITEM_DIMENSIONS failed."))
        exit(1);

    maxWorkItemSizes = (size_t*)malloc(maxDimensions * sizeof(size_t));
    status = clGetDeviceInfo(cl_getDeviceId(), CL_DEVICE_MAX_WORK_ITEM_SIZES, sizeof(size_t) * maxDimensions, (void*)maxWorkItemSizes, NULL);

    if(cl_errChk(status,"clGetDeviceInfo CL_DEVICE_MAX_WORK_ITEM_SIZES failed."))
        exit(1);


    status = clGetDeviceInfo(
        cl_getDeviceId(),
        CL_DEVICE_LOCAL_MEM_SIZE,
        sizeof(cl_ulong),
        (void *)&totalLocalMemory,
        NULL);

    if(cl_errChk(status,"clGetDeviceInfo CL_DEVICE_LOCAL_MEM_SIZE failed."))
        exit(1);
    numBodies = numParticles;
    /*
    * Create and initialize memory objects
    */

    /* Create memory objects for position */
    currPos = clCreateBuffer(cl_getContext(),CL_MEM_READ_WRITE, numBodies * sizeof(cl_float4),0,&status);
    if(cl_errChk(status,"clCreateBuffer failed. (oldPos)"))exit(1);


    /* Initialize position buffer */
    status = clEnqueueWriteBuffer(cl_getCommandQueue(),currPos, 1,0,numBodies * sizeof(cl_float4), pos, 0,0,0);
    if(cl_errChk(status, "clEnqueueWriteBuffer failed. (oldPos)"))
    {
        exit(1);
    }

    /* Create memory objects for position */
    newPos = clCreateBuffer( cl_getContext(), CL_MEM_READ_WRITE, numBodies * sizeof(cl_float4),0, &status);
    if(cl_errChk(status,"clCreateBuffer failed. (newPos)"))
    {
        exit(1);
    }

    /* Create memory objects for velocity */
    currVel = clCreateBuffer( cl_getContext(), CL_MEM_READ_WRITE, numBodies * sizeof(cl_float4), 0, &status);
    if(cl_errChk(status,"clCreateBuffer failed. (oldVel)")) exit(1);


    /* Initialize velocity buffer */
    status = clEnqueueWriteBuffer(cl_getCommandQueue(), currVel, 1,0, numBodies * sizeof(cl_float4), vel,0,0, 0);
    if(cl_errChk(status,"clEnqueueWriteBuffer failed. (oldVel)")) exit(1);


    /* Create memory objects for velocity */
    newVel = clCreateBuffer(cl_getContext(), CL_MEM_READ_ONLY, numBodies * sizeof(cl_float4), 0, &status);
    if(cl_errChk(status,"clCreateBuffer failed. (newVel)"))  exit(1);

    cl_program program;
    if(kernel_type == BASIC)
        program = cl_CompileProgram("../NBody_Kernels_basic.cl",NULL,0);
    else if(kernel_type == REUSE)
        program = cl_CompileProgram("../NBody_Kernels.cl",NULL,0);
    else if(kernel_type == UNROLL)
        program = cl_CompileProgram("../NBody_Kernels_unroll_2.cl",NULL,0);
    else
    {
        printf("Kernel Type Unknown...Exiting\n");
        exit(1);
    }
    /* get a kernel object handle for a kernel with the given name */
    kernel = clCreateKernel( program, "nbody_sim", &status);
    if(cl_errChk(status,"clCreateKernel failed."))  exit(1);


    return SUCCESS;
}


int NBody::setupCLKernels()
{
    cl_int status;

    /* Set appropriate arguments to the kernel */

    /* Particle positions */
    status = clSetKernelArg(kernel,0, sizeof(cl_mem),(void*)&currPos);
    /* Particle velocity */
    status |= clSetKernelArg(kernel,1,sizeof(cl_mem),(void *)&currVel);
    /* numBodies */
    status |= clSetKernelArg(kernel,2,sizeof(cl_int),(void *)&numBodies);
    /* time step */
    status |= clSetKernelArg(kernel,3,sizeof(cl_float), (void *)&delT);
    /* upward Pseudoprobability */
    status |= clSetKernelArg(kernel,4,sizeof(cl_float),(void *)&espSqr);
    /* local memory */
    status = clSetKernelArg(kernel,5,GROUP_SIZE * 4 * sizeof(float),NULL);
    /* Particle positions */
    status |= clSetKernelArg(kernel,6,sizeof(cl_mem),(void*)&newPos);
    /* Particle velocity */
    status |= clSetKernelArg(kernel,7,sizeof(cl_mem),(void *)&newVel);
    if(cl_errChk(status, "clSetKernelArg failed. (unewPos)")) exit(1);


    status = clGetKernelWorkGroupInfo(kernel,
        cl_getDeviceId(),
        CL_KERNEL_LOCAL_MEM_SIZE,
        sizeof(cl_ulong),
        &usedLocalMemory,
        NULL);
    if(cl_errChk(status,"clGetKernelWorkGroupInfo CL_KERNEL_LOCAL_MEM_SIZE failed.")) exit(1);


    if(usedLocalMemory > totalLocalMemory)
    {
        std::cout << "Unsupported: Insufficient local memory on device" <<
            std::endl;
        exit(1);
    }

    /* Check group size against group size returned by kernel */
    status = clGetKernelWorkGroupInfo(kernel,
        cl_getDeviceId(),
        CL_KERNEL_WORK_GROUP_SIZE,
        sizeof(size_t),
        &kernelWorkGroupSize,
        0);
    if(cl_errChk(status,"clGetKernelWorkGroupInfo CL_KERNEL_COMPILE_WORK_GROUP_SIZE failed.")) exit(1);


    if(groupSize > kernelWorkGroupSize)
    {

        std::cout << "Out of Resources!" << std::endl;
        std::cout << "Group Size specified : " << groupSize << std::endl;
        std::cout << "Max Group Size supported on the kernel : "
                  << kernelWorkGroupSize << std::endl;
        std::cout << "Falling back to " << kernelWorkGroupSize << std::endl;

        groupSize = kernelWorkGroupSize;
    }

    return SUCCESS;
}

int NBody::runCLKernels()
{
    cl_int status;
    cl_event events[1];

    /*
    * Enqueue a kernel run call.
    */
    size_t globalThreads[] = {numBodies};
    size_t localThreads[] = {groupSize};

    if(localThreads[0] > maxWorkItemSizes[0] ||
       localThreads[0] > maxWorkGroupSize)
    {
        std::cout << "Unsupported: Device"
            "does not support requested number of work items.";
        exit(1);
    }
    cl_int event_status;
    cl_event event_timer;
    cl_ulong starttime, endtime;

    status = clEnqueueNDRangeKernel( cl_getCommandQueue(), kernel, 1,
        NULL, globalThreads, localThreads,
        0, NULL, &event_timer);



    if(cl_errChk(status,"clEnqueueNDRangeKernel failed.")) exit(1);
    status = clFinish(cl_getCommandQueue());
    if(cl_errChk(status, "clFinish failed.")) exit(1);

    event_status = clGetEventProfilingInfo(event_timer,
                    CL_PROFILING_COMMAND_START  ,
                    sizeof(cl_ulong), &starttime, NULL);

    event_status = clGetEventProfilingInfo(event_timer,
                    CL_PROFILING_COMMAND_END  ,
                    sizeof(cl_ulong), &endtime, NULL);
    if(cl_errChk(event_status, "Error in Get End time\n")) exit(1);

	unsigned long elapsed =  (unsigned long)(endtime - starttime);
	printf("Kernel Execution\t%ld ns \t %ld ms \n",elapsed,elapsed/(1000*1000));

    /* Copy data from new to old */
    status = clEnqueueCopyBuffer(cl_getCommandQueue(), newPos, currPos, 0, 0, sizeof(cl_float4) * numBodies, 0, 0, 0);
    if(cl_errChk(status, "clEnqueueCopyBuffer failed.(newPos->oldPos)")) exit(1);

    status = clEnqueueCopyBuffer(cl_getCommandQueue(), newVel, currVel, 0, 0, sizeof(cl_float4) * numBodies, 0, 0, 0);
    if(cl_errChk(status,"clEnqueueCopyBuffer failed.(newVel->oldVels)"))exit(1);
    status = clFinish(cl_getCommandQueue());
    if(cl_errChk(status,"clFinish failed.")) exit(1);


    /* Enqueue readBuffer*/
    status = clEnqueueReadBuffer(
        cl_getCommandQueue(),
        currPos,
        CL_TRUE,
        0,
        numBodies* sizeof(cl_float4),
        pos,
        0,
        NULL,
        &events[0]);
    if(cl_errChk( status, "clEnqueueReadBuffer failed.")) exit(1);

    /* Wait for the read buffer to finish execution */
    status = clWaitForEvents(1, &events[0]);
    if(cl_errChk(status,"clWaitForEvents failed.")) exit(1);

    clReleaseEvent(events[0]);

    return SUCCESS;
}

/*
* n-body simulation on cpu
*/
void
NBody::nBodyCPUReference()
{
    //Iterate for all samples
    for(int i = 0; i < numBodies; ++i)
    {
        int myIndex = 4 * i;
        float acc[3] = {0.0f, 0.0f, 0.0f};
        for(int j = 0; j < numBodies; ++j)
        {
            float r[3];
            int index = 4 * j;

            float distSqr = 0.0f;
            for(int k = 0; k < 3; ++k)
            {
                r[k] = refPos[index + k] - refPos[myIndex + k];

                distSqr += r[k] * r[k];
            }

            float invDist = 1.0f / sqrt(distSqr + espSqr);
            float invDistCube =  invDist * invDist * invDist;
            float s = refPos[index + 3] * invDistCube;

            for(int k = 0; k < 3; ++k)
            {
                acc[k] += s * r[k];
            }
        }

        for(int k = 0; k < 3; ++k)
        {
            refPos[myIndex + k] += refVel[myIndex + k] * delT + 0.5f * acc[k] * delT * delT;
            refVel[myIndex + k] += acc[k] * delT;
        }
    }
}

int
NBody::initialize()
{
    /* Call base class Initialize to get default configuration */

    //    streamsdk::Option *num_particles = new streamsdk::Option;
    if(!num_particles)
    {
        std::cout << "error. Failed to allocate memory (num_particles)\n"<<num_particles;
//        exit(1);
    }

    return SUCCESS;
}

int NBody::setup()
{

    cl_init_context();

    if(setupNBody() != SUCCESS)
        exit(1);

    if(setupCL() != SUCCESS)
        exit(1);

    return SUCCESS;
}

int NBody::verifyResults()
{
    /* reference implementation
    * it overwrites the input array with the output
    */
    if(verify)
    {
        printf("Verify with %d\n",numBodies);
        refPos = (cl_float*)malloc(numBodies * sizeof(cl_float4));
        if(refPos == NULL)
        {
            printf("Failed to allocate host memory. (refPos)");
            exit(1);
        }

        refVel = (cl_float*)malloc(numBodies * sizeof(cl_float4));
        if(refVel == NULL)
            exit(1);


        memcpy(refPos, initPos, 4 * numBodies * sizeof(cl_float));
        memcpy(refVel, initVel, 4 * numBodies * sizeof(cl_float));

        for(int i = 0; i < iterations; ++i)
        {
            nBodyCPUReference();
        }

        /* compare the results and see if they match */
        if(compare(pos, refPos, 4 * numBodies, 0.001))
        {
            std::cout << "Passed!\n";
            return 1;
        }
        else
        {
            std::cout << "Failed!\n";
            exit(1);
        }
    }

    return 1;
}


bool NBody::compare(const float *refData, const float *data,
                    const int length, const float epsilon)
{
    float error = 0.0f;
    float ref = 0.0f;


    for(int i = 1; i < length; ++i)
    {
        float diff = refData[i] - data[i];
        error += diff * diff;
        ref += refData[i] * refData[i];
    }

    float normRef =::sqrtf((float) ref);
    if (::fabs((float) ref) < 1e-7f) {
        printf("error\n");
        return false;
    }
    float normError = ::sqrtf((float) error);
    error = normError / normRef;

//    printf("error \t %f epsilon %f\n",error,epsilon);

    return error < epsilon;
}




int NBody::cleanup()
{
    /* Releases OpenCL resources (Context, Memory etc.) */
    cl_int status;
    cl_cleanup();


    status = clReleaseMemObject(currPos);
    if(cl_errChk(status, "clReleaseMemObject failed.")) exit(1);
    status = clReleaseMemObject(currVel);
    if(cl_errChk(status,"clReleaseMemObject failed.")) exit(1);


    return SUCCESS;
}

NBody::~NBody()
{
    /* release program resources */
    if(initPos)
    {
        free(initPos);
        initPos = NULL;
    }

    if(initVel)
    {
        free(initVel);
        initVel = NULL;
    }

    if(pos)
    {
#if defined (_WIN32)
        _aligned_free(pos);
#else
        free(pos);
#endif
        pos = NULL;
    }
    if(vel)
    {
#if defined (_WIN32)
        _aligned_free(vel);
#else
        free(vel);
#endif
        vel = NULL;
    }

    if(devices)
    {
        free(devices);
        devices = NULL;
    }

    if(refPos)
    {
        free(refPos);
        refPos = NULL;
    }

    if(refVel)
    {
        free(refVel);
        refVel = NULL;
    }

    if(maxWorkItemSizes)
    {
        free(maxWorkItemSizes);
        maxWorkItemSizes = NULL;
    }
}




/**
* @brief OpenGL display function
*/
void displayfunc()
{

    glClearColor(0.0 ,0.0, 0.0, 0.0);
    glClear(GL_COLOR_BUFFER_BIT);
    glClear(GL_DEPTH_BUFFER_BIT);

    glPointSize(1.0);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE);
    glEnable(GL_BLEND);
    glDepthMask(GL_FALSE);

    glColor3f(1.0f,0.6f,0.0f);

    //Calling kernel for calculatig subsequent positions
     ((NBody*)me)->runCLKernels();

    glBegin(GL_POINTS);
    for(int i=0; i < numBodies; ++i)
    {
        //divided by 300 just for scaling
        glVertex3d(pos[i*4+ 0]/300, pos[i*4+1]/300, pos[i*4+2]/300);
    }
    glEnd();

    glFlush();
    glutSwapBuffers();
}

/* keyboard function */
void keyboardFunc(unsigned char key, int mouseX, int mouseY)
{
    switch(key)
    {
        /* If the user hits escape or Q, then exit */
        /* ESCAPE_KEY = 27 */
    case 27:
    case 'q':
    case 'Q':
        {
            if( ((NBody*)me)->cleanup() != SUCCESS)
                exit(1);
            else
                exit(0);
        }
    default:
        break;
    }
}

