#include "daemon.h"
#include "daemon_exception.h"
#include "parser.h"
#include "parser_exception.h"
#include "logger.h"

#include <iostream>
#include <fstream>
#include <strstream>
#include <string>

extern "C"{
     //for fork()
#include <unistd.h>
     //for perror();
#include <stdio.h>
     //for sys_errlist
#include <errno.h>
     //for stat()
#include <sys/stat.h>
     //for uid_t
#include <sys/types.h>
}

const static int MAXFD = 64;

void daemon::daemonize_me(const char * prog_name)
{
     //copied directly from Stevens- Unix Network Programming, p. 336
     int pid;

     if ((pid = fork()) != 0){
	  //check if fork failed
	  if (pid == -1){
	       //we can still write to stderr, since we're still attached
	       //to a tty
	       std::cerr<<"daemon::daemonize_me: first fork() failed."<<endl;
	       perror("error returned: ");
	       return;
	  }
	  //if pid != 0 && pid != -1 we're in the parent now, so we exit
	  exit(0);
     }

     //child continues
     setsid(); //can't fail. man 2 setsid.

     //now open the logger::log for messages, so we can
     //report if we fail

     if ((pid = fork()) != 0){
	  //check if fork failed
	  if (pid == -1){
	       //we're no longer attached to a tty, so let's write to syslog
	       logger::log("daemon::daemonize_me second fork() failed");
	       strstream stream;
	       stream<<"error is: "<<sys_errlist[errno]<<endl;
	       logger::log(stream.str());
	  }
	  //if pid != 0 && pid != -1 we're in the parent now, so we exit
	  exit(0);
     }

     //second child continues

     chdir("/");

//     umask(0);
     
     //close all open file descriptors
     for (int i = 0; i < MAXFD; ++i){
	  close(i);
     }
}


void daemon::load_config_file(const char * config_file_name)
{
     //first of all, make some sanity checks on the file

     //preferably, that user will be root, but we relax that requirement for
     //development
     struct stat stat_info;
     if (stat(config_file_name, &stat_info)==-1){
	  logger::log("load_config_file::stat failed, with error:");
	  logger::log(sys_errlist[errno]);
	  throw daemon_exception("stat failed");
     }

     //the config file is a regular file
     if (!S_ISREG(stat_info.st_mode)){
	  logger::log("load_config_file: file is not a regular file: ");
	  logger::log(config_file_name);
	  throw daemon_exception("file is not a regular file");
     }
     //the config file is owned by the current user
     if (stat_info.st_uid != geteuid()){
	  logger::log("load_config_file: file is not owned by the current euid");
	  logger::log(config_file_name);
	  throw daemon_exception("file not owned by current euid");
     }

     //the config file is only writeable by the current user. 
/*     if ((stat_info.st_mode & (S_IRUSR | S_IWUSR)) != (S_IRUSR & S_IWUSR)){
	  log("load_config_file: wrong permissions on config_file");
	  log(config_file_name);
	  throw daemon_exception("file has wrong permission");
	  }*/

     //now, that we've verified the file is usable by us, let's load
     //its contents into a buffer.
     ifstream in(config_file_name);
     in>>_config_data.rdbuf();
}

void daemon::parse_config_file()
{
     //check wether we've uploaded the config data
     if (_config_data.eof()){
	  logger::log("parse_config_file: no config data available to parser");
	  throw daemon_exception("no config data available to parser");
     }
     //create a new parser
     parser p(_config_data);
     //start parsing the config data
     try{
	  p.parse(_rules);
     }catch(parser_exception& pex){
	  cerr<<pex;
     }
}



