#include <iostream>
#include <fstream>
#include <strstream>
#include <list>

extern"C"{
#include <unistd.h>
#include <errno.h>
#include <string.h>
     //for stat()
#include <sys/stat.h>
     //for uid_t
#include <sys/types.h>
}

#include "rule.h"
#include "logger.h"
#include "parser.h"

const char CONFIG_FILE_NAME[] = "../daemon.conf";


/* forward declarations */
bool load_config_file(const char * config_file_name,
		      strstream& data);
bool parse_config_file(strstream& data, list<rule>& rules);



int main(int argc, char ** argv)
{
     strstream data;
     if (!load_config_file(CONFIG_FILE_NAME, data)){
	  logger::log("failed to read the config file. quitting.");
	  return -1;
     }
     //the list of rules
     list<rule> rules;

     if (!parse_config_file(data, rules)){
	  logger::log("failed to parse the config file. quitting.");
	  return -1;
     }

     //make sure all rules are valid and print them.
     cerr<<"the following rules were parsed: "<<endl;
     list<rule>::const_iterator it = rules.begin();
     for (; it != rules.end(); ++it){
	  if (!(*it).is_valid()){
	       cerr<<"*** invalid rule found ***"<<endl;
	  }
	  cerr<<"*** rule: "<<(*it).get_rule_name()<<" ***"<<endl;
	  cerr<<*it<<endl<<endl;

     }

     //do something with the rules. 
     return 0;
}


bool load_config_file(const char * config_file_name,
		     strstream& data)
{
     //first of all, make some sanity checks on the file

     //preferably, that user will be root, but we relax that requirement for
     //development
     struct stat stat_info;
     if (stat(config_file_name, &stat_info)==-1){
	  logger::log("load_config_file::stat failed, with error:");
	  logger::log(strerror(errno));
	  return false;
     }

     //the config file is a regular file
     if (!S_ISREG(stat_info.st_mode)){
	  logger::log("load_config_file: file is not a regular file: ");
	  logger::log(config_file_name);
	  return false;
     }
     //the config file is owned by the current user
     if (stat_info.st_uid != geteuid()){
	  logger::log("load_config_file: file is not owned by the current euid");
	  logger::log(config_file_name);
	  return false;
     }

     //the config file is only writeable by the current user. 
/*     if ((stat_info.st_mode & (S_IRUSR | S_IWUSR)) != (S_IRUSR & S_IWUSR)){
	  log("load_config_file: wrong permissions on config_file");
	  log(config_file_name);
	  return false;
	  }*/

     //now, that we've verified the file is usable by us, let's load
     //its contents into a buffer.
     ifstream in(config_file_name);
     in>>data.rdbuf();
     return true;
}

bool parse_config_file(strstream& data, list<rule>& rules)
{
     //create a new parser
     parser p(data);
     //start parsing the config data
     return p.parse(rules);
}





