#include "rule.h"
#include "validity_checks.h"
#include "logger.h"

#include <iostream>
#include <list>
#include <strstream>

rule_identifier rule::last_rule_id = 0;

ostream& operator<<(ostream& os, const rule& r)
{
     os<<"rule: "<<r._rule_id<<endl;
     os<<"\tsyscall: "<<r._syscall_id<<endl;
     list<param_conditional*>::const_iterator prit = r._param_cond.begin();
     for(;prit != r._param_cond.end();++prit) os<<**prit;
     list<process_conditional*>::const_iterator pcit = r._proc_cond.begin();
     for(;pcit != r._proc_cond.end();++pcit) os<<**pcit;

     os<<"\taction: ";
     switch (r._act){
     case (LOG):os<<"LOG"<<endl;
	  break;
     case (SUSPEND):os<<"SUSPEND"<<endl;
	  break;
     case (FAIL):os<<"FAIL"<<endl;
	  break;
     case(KILL):os<<"KILL"<<endl;
	  break;
     default:
	  os<<"unknown action!"<<endl;
	  break;
     }
     return os;
}

rule::rule():_rule_id(++last_rule_id),_syscall_id(0),
	     _act(LAST_ACTION), _cond_match(LAST_COND_MATCH){}

rule::rule(const rule& r)
{
     _rule_id = r._rule_id;
     _syscall_id = r._syscall_id;
     _rule_name = r._rule_name;
     _act = r._act;
     //copy the lists of pointers
     list<param_conditional*>::const_iterator prit = r._param_cond.begin();
     for (; prit != r._param_cond.end(); ++prit){
	  _param_cond.push_back(new param_conditional(**prit));
     }
     
     list<process_conditional*>::const_iterator pcit = r._proc_cond.begin();
     for (; pcit != r._proc_cond.end(); ++pcit){
	  _proc_cond.push_back(new process_conditional(**pcit));
     }
}

rule::~rule()
{
     list<param_conditional*>::iterator prit = _param_cond.begin();
     for (; prit != _param_cond.end(); ++prit){
	  delete *prit;
     }
     
     list<process_conditional*>::iterator pcit = _proc_cond.begin();
     for (; pcit != _proc_cond.end(); ++pcit){
	  delete *pcit;
     }
}


void rule::set_syscall_identifier(const string& str)
{
     if (is_syscall_identifier(str)){
	  //TODO: we need a mapping from syscall names to numbers
	  logger::log("***BUG: rule::set_syscall_identifier()");
	  logger::log("string to syscall mapping not implemented yet!");
	  _syscall_id = 1;
     }else{
	  cerr<<"set_syscall_identifier: '"<<str<<"'is not a valid syscall id"<<endl;
     }
}

inline syscall_identifier rule::get_syscall_identifier() const
{
     return _rule_id;
}

void rule::set_action(const string& str)
{
     if (is_action(str)){
	  if (str=="LOG"){
	       _act = LOG;
	  }else if (str=="SUSPEND"){
	       _act = SUSPEND;
	  }else if (str=="FAIL"){
	       _act = FAIL;
	  }else if (str=="KILL"){
	       _act = KILL;
	  }
     }else{
	  cerr<<"set_action: '"<<str<<" is not a valid action!"<<endl;
     }
}

param_conditional* rule::get_param_conditional()
{
     param_conditional* pcond = new param_conditional();
     if (!pcond)
	  return NULL;
     _param_cond.push_back(pcond);
     return pcond;
}

process_conditional* rule::get_process_conditional()
{
     process_conditional* pcond = new process_conditional();
     if (!pcond)
	  return NULL;
     _proc_cond.push_back(pcond);
     return pcond;
}

void rule::set_condition_matching(const string& str){
     if (!is_condition_matching(str)){
	  cerr<<"set_condition_matching: '"<<str<<"' is not a valid token"<<endl;
	  return;
     }
     if (str=="ANY"){
	  _cond_match = ANY;
     }else {
	  _cond_match = ALL;
     }
}

bool rule::is_valid() const
{
     bool valid = true;
     if (_syscall_id == 0) {
	  cerr<<"rule '"+_rule_name+"' does not have a valid syscall"<<endl;
	  valid = false;
     }
     if (_rule_name.empty()){
	  cerr<<"rule with id: '"<<_rule_id<<"' has no rule name"<<endl;
	  valid = false;
     }
     if (_act == LAST_ACTION){
	  cerr<<"rule '"+_rule_name+"' does not have a valid action"<<endl;
	  valid = false;
     }
     if (!param_conds_valid()){
	  cerr<<"rule '"+_rule_name+"' has invalid paramaeter condition clasue"<<
	       endl;
	  valid = false;
     }
     if (!proc_conds_valid()){
	  cerr<<"rule '"+_rule_name+"' has invalid process condition clause"<<
	       endl;
	  valid = false;
     }
     if ((_proc_cond.size() != 0) && (_param_cond.size() != 0)){
	  if (_cond_match == LAST_COND_MATCH){
	       cerr<<"rule '"+_rule_name+"' has at least two conditional clauses"<<
		    "and no CONDITIONAL_MATCHING clause"<<endl;
	       valid = false;
	  }
     }
     return valid;
}
     


bool rule::proc_conds_valid() const
{
     bool res = true;
     list<process_conditional*>::const_iterator it = _proc_cond.begin();
     for (; it != _proc_cond.end();++it){
	  if (!(*it)->is_valid()){
	       strstream buf;
	       buf<<**it;
	       logger::log("found an invalid process conditional clause: ");
	       logger::log(buf.str());
	       res = false;
	  }
     }
     return res;
}

bool rule::param_conds_valid() const
{
     bool res = true;
     list<param_conditional*>::const_iterator it = _param_cond.begin();
     for (; it != _param_cond.end();++it){
	  if (!(*it)->is_valid()){
	       strstream buf;
	       buf<<**it;
	       logger::log("found an invalid parameter conditional clause: ");
	       logger::log(buf.str());
	       res = false;
	  }
     }
     return res;
}
