/* This class represents each rule the daemon builds from the config file.
 * @author: mulix
 */

#ifndef _RULE_H_
#define _RULE_H_

#include <list>

//for conditional clauses in the the rule
#include "param_conditional.h"
#include "process_conditional.h"

//for identifying this rule
typedef unsigned int rule_identifier;


//we identify a syscall by its index in the syscall table,
//for the time being.
typedef unsigned short int syscall_identifier;

typedef enum action_t {
     LOG = 0,
     SUSPEND,
     FAIL,
     KILL,
     LAST_ACTION
} action;

typedef enum condition_matching_t{
     ANY = 0,
     ALL,
     LAST_COND_MATCH
};


class rule{
 public:
     //constructor- we give the the rule a unique identifier
     rule();
     //destructor
     ~rule();

     //copy constructor
     rule(const rule& r);

     //is_valid return true if a rule is valid an false otherwise. 
     bool is_valid() const;

     //get method for the rule_id
     rule_identifier get_rule_id() const {return _rule_id;}

     //set get methods for syscall
     void set_syscall_identifier(const string& str);
     syscall_identifier get_syscall_identifier() const;

     //get_param_conditional allocates a new param_conditional object
     //adds it to the rule's list of param_conditionals and returns it. 
     param_conditional* get_param_conditional();

     //get_process_conditional allocates a new process_conditional object
     //adds it to the rule's list of process_conditionals and returns it. 
     process_conditional* get_process_conditional();

     //set get methods for the action
     void set_action(const string& str);
     action get_action(void) const {return _act;}

     //get set methods for the rule name. 
     void set_rule_name(const string& str){_rule_name = str;}
     string get_rule_name() const {return _rule_name;} 

     void set_condition_matching(const string& str);
     condition_matching_t get_condition_matching() const {return _cond_match;}

     //have the rule write itself to an ostream.
     friend ostream& operator<<(ostream& os, const rule& r);

     //module_comm needs to know how we're built so that it can serialize us
     friend class module_comm;
 private:
     //checks whether all of the param conditionals are valid. 
     bool param_conds_valid() const;

     //checks whether all of the proc conditionals are valid. 
     bool proc_conds_valid() const;

     /* members */
     rule_identifier _rule_id;
     string _rule_name;
     syscall_identifier _syscall_id;
     list<param_conditional*> _param_cond;
     list<process_conditional*> _proc_cond;
     action _act;
     condition_matching_t _cond_match;

     /* static members */
     static rule_identifier last_rule_id;
};

#endif //_RULE_H_
