/**********************************************************************
                                                                     
   Copyright (C) 2002,2003 Emil D. Kohn   emild@cs.technion.ac.il     
                                                                     
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 **********************************************************************/


#include <pthread.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <asm/atomic.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <usb.h>

#include "sar.h"
#include "pppd_io.h"


/*#define  DUMP_BUFFERS*/

#ifdef DUMP_BUFFERS
#define LOG_FILE_NAME             "/tmp/adsl.log"
#endif

#define APACHE_MODEM_VENDOR_ID    0x0572
#define APACHE_MODEM_PRODUCT_ID   0xCAFE

/* Place some [un]reasonably high values */
#define TX_TIMEOUT   1000000000  /* milliseconds */
#define RX_TIMEOUT   2000000000

#define TX_BUFFER_SIZE           4096
#define RX_BUFFER_SIZE           (56 + 4096)
#define PPP_FRAME_BUFFER_SIZE    4096


#define MODEM_STATUS_ENDPOINT    0x81
#define MODEM_DATA_IN_ENDPOINT   0x82
#define MODEM_DATA_OUT_ENDPOINT  0x02

#define LINE_STATUS_BUFFER_SIZE  0x100
#define LINE_STATUS_CODE         0x88
#define REPLY                    0x01
#define LINE_STATUS_OFFSET       0x24
#define RX_SN_RATIO_OFFSET       0x54

#define MIN_LINE_UP_READS        5

#define PPPD_EXEC                "/usr/sbin/pppd"

/* The firmware_data.h/setup_data.h and firmware_info.h/setup_info.h files 
   are auto-generated from the usbsnoop.log file by the c_array.pl script */

#include "firmware_data.h"
#include "setup_data.h"
#include "line_status_data.h"

#ifdef DUMP_BUFFERS
static pthread_mutex_t  log_fp_mutex;
static FILE* log_fp;
#endif

struct transfer_info {
  int endpoint;
  char* data;
  int size;
  char* name; 
};


typedef enum {
  ADSL_STATUS_DOWN = 0,
  ADSL_STATUS_ATTEMPTING_TO_ACTIVATE,
  ADSL_STATUS_TRAINING,
  ADSL_STATUS_CHANNEL_ANALYSIS,
  ADSL_STATUS_EXCHANGE,
  ADSL_STATUS_UP
} ADSL_LINE_STATUS;


struct transfer_info  firmware_transfer[] = {

#include "firmware_info.h"

};

  
#define FIRMWARE_TRANSFER_SIZE (sizeof(firmware_transfer)/sizeof(firmware_transfer[0]))


struct transfer_info setup_transfer[] = {

#include "setup_info.h"

};

#define SETUP_TRANSFER_SIZE (sizeof(setup_transfer)/sizeof(setup_transfer[0]))


struct transfer_info line_status_transfer[] = {

#include "line_status_info.h"

};

#define LINE_STATUS_TRANSFER_SIZE  (sizeof(line_status_transfer)/sizeof(line_status_transfer[0]))


struct usb_device* dev;
struct usb_dev_handle* dev_handle;

int from_pppd_fd;
int to_pppd_fd;

static pthread_t pppd_to_modem_thread;
static pthread_t modem_to_pppd_thread;
static pthread_t line_status_reader_thread;
  pid_t pppd_pid;
static int ifc_num;

static atomic_t adsl_line_status;


const char* adsl_status_str(ADSL_LINE_STATUS status)
{
  switch (status) {
  case ADSL_STATUS_DOWN:
    return "DOWN";

  case ADSL_STATUS_ATTEMPTING_TO_ACTIVATE:
    return "ATTEMPTING TO ACTIVATE";

  case ADSL_STATUS_TRAINING:
    return "TRAINING";

  case ADSL_STATUS_CHANNEL_ANALYSIS:
    return "CHANNEL ANALYSIS";

  case ADSL_STATUS_EXCHANGE:
    return "EXCHANGE";

  case ADSL_STATUS_UP:
    return "UP";

  default:
    return "UKNOWN";
  }
}


struct usb_device* find_device(u_int16_t vendor,
			       u_int16_t product)
{
  struct usb_bus* bus;
  struct usb_device* dev;

  for (bus = usb_busses; bus != NULL; bus = bus->next) {
    for (dev = bus->devices; dev != NULL; dev = dev->next)
      if (dev->descriptor.idVendor == vendor &&
	  dev->descriptor.idProduct == product)
	return dev;
  }
   
  return NULL;
}


void dump_configurations(struct usb_device* dev,
			 struct usb_dev_handle* dev_handle)
{
  struct usb_config_descriptor* cfg;
  int num_configurations = dev->descriptor.bNumConfigurations;
  int i;

  printf("The device supports %d configurations\n",num_configurations);
  
  for (i=0; i<num_configurations; i++) {
    printf ("Configuration #%d\n",i);
    for (cfg = &dev->config[0]; i<num_configurations; i++) {
      char buffer[256] = "Unknown";
      int rc;
      int j;
      int num_ifc = cfg->bNumInterfaces;

      if (cfg->iConfiguration != 0) {
	rc = usb_get_string_simple(dev_handle,cfg->iConfiguration,buffer,sizeof(buffer));
	assert(rc > 0);
	buffer[sizeof(buffer)-1] = '\0'; /* Just to be sure */
	printf("Configuration Name: %s\n",buffer);
      } else {
	printf("Configuration Name: <UNKNOWN>\n");
      }
      printf("Configuration ID: %d\n",cfg->bConfigurationValue);
      printf("Power: %dmA\n",cfg->MaxPower*2);
      printf("\tNumber of interfaces: %d\n",num_ifc);

      for (j=0; j<num_ifc; j++) {
	struct usb_interface* ifc = &cfg->interface[j];
	int k;
	
	int num_settings = ifc->num_altsetting;

	printf("\tInterface #%d\n",j);
	printf("\t\tNumber of settings:%d\n",num_settings);

	for (k=0; k<num_settings;k++) {
	  struct  usb_interface_descriptor* setting;

	  setting = &ifc->altsetting[k];
	  printf("\t\tSetting# %d\n",k); 
#if 0
	  rc = usb_get_string_simple(dev_handle,setting->iInterface,buffer,sizeof(buffer));
	  assert(rc > 0);
	  printf("\t\tSetting Name: %s\n",buffer);
#else
	  printf("\t\tSetting Name: <UNKNOWN>\n");
#endif
	  printf("\t\tSetting ID: %d\n",setting->bInterfaceNumber);
	}
	printf("======================================================\n");
      }
      printf("++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n");
    }
    printf("---------------------------------------------------------------\n");
  }
}


static int  reset_pipes(struct usb_device* dev,
			struct usb_dev_handle* dev_handle)
{
  struct usb_interface_descriptor* ifc;
  struct usb_endpoint_descriptor* ep;
  int i;
  int rc;
  int ret_val = 0;

  ifc = &dev->config[0].interface[0].altsetting[0];

  printf("Resetting endpoints.\n");

  for (i=0; i<ifc->bNumEndpoints; i++) {
    ep = &ifc->endpoint[i];

    printf("                                                      \r");
    printf("%d [0x%02x]...",i+1,ep->bEndpointAddress);
    fflush(stdout);

    rc = usb_resetep(dev_handle,ep->bEndpointAddress);
    if (rc >= 0) {
      printf ("Success.\r");
    } else {
      printf("Failed [%s].\n",usb_strerror());
      ret_val = -1;
      break;
    }
  }
  printf ("\n");
  return ret_val;
}


static int transfer_data(struct usb_dev_handle* dev_handle,
			 struct transfer_info transfers[],
			 int num_transfers)
{
  struct transfer_info* trn;
  int i;
  int rc;

  printf("Transfering data\n");

  for (i=0; i<num_transfers; i++) {
    trn = &transfers[i];
    printf("                                                                 \r");
    printf ("Endpoint 0x%02x, %s...",trn->endpoint, trn->name);
    fflush(stdout);
    rc = usb_bulk_write(dev_handle, trn->endpoint, trn->data,trn->size, TX_TIMEOUT);

    if (rc >= 0) {
      printf("Success.\r");
      fflush(stdout);
    } else {
      printf("Failed [%s].\n",usb_strerror());
      goto err_bulk_write;
    }
  }
  printf("\n");
  return 0;

 err_bulk_write:
  return -1;
}


#ifdef DUMP_BUFFERS
void dump_buffer(FILE* fp, const char* buff, int size)
{
  int i;

  for (i=0;i<size;i++) {
    if ((i % 16) == 0) {
      fprintf(fp,"%08x:",i);
    }
    fprintf(fp," %02x",(int)(unsigned char)buff[i]);
    if ((i % 16) == 15) {
      fprintf(fp,"\n");
    }
  }
  if ((i % 16) != 0) 
    fprintf(fp,"\n"); 
#if 0
  fprintf(fp,"\n\n*****************************************************\n");
  
  for (i=0;i<size;i++) {
    fprintf(fp,"%02x ",(int)(unsigned char)buff[i]);
  }
  fprintf(fp,"\n\n");
  
  fprintf(fp,"############################################################\n\n");
#endif
  fflush(fp);
}
#endif





void *line_status_reader(void* arg)
{
  u_int8_t buff[4096];
  int num_ups = 0;
  int rc;

  /*  pthread_detach(pthread_self());*/
  
  rc = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
  if (rc != 0) 
    goto error;

  rc = pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);
  if (rc != 0)
    goto error;

  for (;;) {

    rc = usb_bulk_read(dev_handle, MODEM_STATUS_ENDPOINT, buff, sizeof(buff), RX_TIMEOUT);

    if (rc >= 0) {
      if (rc == LINE_STATUS_BUFFER_SIZE && 
	  buff[0] == LINE_STATUS_CODE && 
	  buff[1] == REPLY) {
	ADSL_LINE_STATUS line_status = buff[LINE_STATUS_OFFSET];
	
	printf("                                                                 \r");
	printf("Line status:  %s.", adsl_status_str(line_status));

	if (line_status == ADSL_STATUS_UP) {
	  printf(" RX S/N ratio: %d.%02ddB",
		 buff[RX_SN_RATIO_OFFSET + 1],
		 buff[RX_SN_RATIO_OFFSET] * 100/256);
	  if (++num_ups >= MIN_LINE_UP_READS) {
	    atomic_set(&adsl_line_status,line_status);
	    num_ups = 0;
	  }
	} else {
	  atomic_set(&adsl_line_status,line_status);
	}
	printf("\r");
      }
    } else {
      printf("Read failed from endpoint 0x%02x [%s]\n",MODEM_STATUS_ENDPOINT,usb_strerror());
    }
    fflush(stdout);
  }
  return NULL;
 error:
#ifdef DUMP_BUFFERS
  fprintf(log_fp,"FATAL ERROR: line status reader thread exiting\n");
#endif
  return NULL;
}





int start_pppd(int argc, char* argv[],
	       int* from_pppd_fd, int* to_pppd_fd)
{
  int from_pppd[2];
  int to_pppd[2];
  int rc;


  rc = pipe(from_pppd);

  if (rc < 0) {
    goto err_pipe_from_pppd;
  }

  rc = pipe(to_pppd);

  if (rc < 0) {
    goto err_pipe_to_pppd;
  }

  pppd_pid = fork();

  if (pppd_pid < 0)
    goto err_fork_pppd;

  if (pppd_pid == 0) {
    dup2(to_pppd[0],0);
    dup2(from_pppd[1],1);
    close(from_pppd[0]);
    close(from_pppd[1]);
    close(to_pppd[0]);
    close(to_pppd[1]);

    execv(PPPD_EXEC,argv);
    /*"921600", "notty", "nodeflate", "nobsdcomp", "user", "emild@IActcom", 
      (char*)NULL,(char*)NULL);*/

    fprintf(stderr,"Cannot start pppd executable [%s]\n",strerror(errno));

    kill(getppid(),SIGTERM);

    exit(1);
  }

  close(from_pppd[1]);
  close(to_pppd[0]);

  *from_pppd_fd = from_pppd[0];
  *to_pppd_fd = to_pppd[1];

  return 0;

 err_fork_pppd:
  close(to_pppd[0]);
  close(to_pppd[1]);
 err_pipe_to_pppd:
  close(from_pppd[0]);
  close(from_pppd[1]);
 err_pipe_from_pppd:
  return -1;
}



static int cell_buffer_fill(u8* cell_buffer, int cell_buffer_size)
{
  int nbytes;

  nbytes = usb_bulk_read(dev_handle, MODEM_DATA_IN_ENDPOINT,
			 cell_buffer,cell_buffer_size,RX_TIMEOUT);

  if (nbytes < 0) {
    fprintf(stderr,"Error reading from endpoint 0x82: %s\n",usb_strerror());
  }
  assert(nbytes >= 0);

  /*  assert(nbytes > 0);*/
  
#ifdef DUMP_BUFFERS
  pthread_mutex_lock(&log_fp_mutex);
  fprintf(log_fp,">>>Attempted to read %d bytes from USB<<<\n",cell_buffer_size);
  dump_buffer(log_fp,cell_buffer,nbytes);
  fprintf(log_fp,"<<Read %d bytes from USB>>>\n",nbytes);
  pthread_mutex_unlock(&log_fp_mutex);
#endif

  return nbytes;
}



static void* modem_to_pppd(void* arg)
{
  u8 cell_buffer[RX_BUFFER_SIZE];
  u8 ppp_frame[PPP_FRAME_BUFFER_SIZE];
  int cell_buffer_size;
  u8* head = cell_buffer;
  u8* tail = cell_buffer;
  int rc;
  
  rc = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
  if (rc != 0) 
    goto error;

  rc = pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);
  if (rc != 0)
    goto error;


  for (;;) {
    int ppp_frame_size;
 
    ppp_frame_size = reassembly(cell_buffer,sizeof(cell_buffer),
				&head,&tail,
				ppp_frame,sizeof(ppp_frame),
				cell_buffer_fill);
    if (ppp_frame_size < 0) 
      goto error; 
    
#ifdef DUMP_BUFFERS
    pthread_mutex_lock(&log_fp_mutex);
    fprintf(log_fp,">>>>>MODEM --> PPPD<<<<<\n");
    fprintf(log_fp,">>>Writing PPP frame<<<\n");
    dump_buffer(log_fp,ppp_frame,ppp_frame_size);
    fprintf(log_fp,"<<<PPP frame written>>>\n");
    fprintf(log_fp,"<<<<<MODEM --> PPPD>>>>>\n\n");
    pthread_mutex_unlock(&log_fp_mutex);
#endif

    rc = write_to_pppd(to_pppd_fd,ppp_frame,ppp_frame_size,0xFFFFFFFF);
    if (rc < 0) 
      goto error;
  }

  return NULL;
  
 error:
#ifdef DUMP_BUFFERS
  fprintf(log_fp,"FATAL ERROR: rx thread exiting\n");
#endif
  return NULL;
}



static void* pppd_to_modem(void* arg)
{
  int ppp_frame_size;
  u8 cell_buffer[TX_BUFFER_SIZE];
  u8 ppp_frame[PPP_FRAME_BUFFER_SIZE];
  int cell_buffer_size;
  int rc;

  rc = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
  if (rc != 0) 
    goto error;

  rc = pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);
  if (rc != 0)
    goto error;

  for (;;) {
    ppp_frame_size = read_from_pppd(from_pppd_fd,ppp_frame,sizeof(ppp_frame));

#ifdef DUMP_BUFFERS
    pthread_mutex_lock(&log_fp_mutex);
    fprintf(log_fp,">>>>>PPPD --> MODEM<<<<<\n");
    fprintf(log_fp,">>>Before segmentation<<<\n");
    dump_buffer(log_fp,ppp_frame,ppp_frame_size);
    fprintf(log_fp,"<<<Before segmentation>>>\n");
    pthread_mutex_unlock(&log_fp_mutex);
#endif

    if (ppp_frame_size < 0)
      goto error;

    cell_buffer_size = segmentation(ppp_frame,ppp_frame_size,sizeof(ppp_frame),
				    cell_buffer,sizeof(cell_buffer));

    if (cell_buffer_size < 0)
      goto error;

#ifdef DUMP_BUFFERS
    pthread_mutex_lock(&log_fp_mutex);
    fprintf(log_fp,">>>Cell  buffer<<<\n");
    dump_buffer(log_fp,cell_buffer,cell_buffer_size);
    fprintf(log_fp,"<<<Cell buffer>>>\n");
    fprintf(log_fp,"<<<<<PPPD --> MODEM>>>>>\n\n");
    pthread_mutex_unlock(&log_fp_mutex);
#endif
    rc = usb_bulk_write(dev_handle,MODEM_DATA_OUT_ENDPOINT,cell_buffer,cell_buffer_size,TX_TIMEOUT);

    if (rc < 0)
      goto error;

    rc = usb_bulk_write(dev_handle,MODEM_DATA_OUT_ENDPOINT,cell_buffer,0,TX_TIMEOUT);

#ifdef DUMP_BUFFERS
    /*    fprintf(log_fp,"tx zero bytes write returns %d\n",rc);*/
#endif
  }

  return NULL;

 error:
#ifdef DUMP_BUFFERS
  fprintf(log_fp,"FATAL ERROR: tx thread exiting\n");
#endif
  return NULL;
}





static int start_pppd_io_threads(u8 vpi, u16 vci)
{
  int rc;
  u8 start_cmd[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x01, 0x00, 
    0x00, 0x00, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
    0xff, 0xff, 0xff, 0x6a, 0x6a, 0x6a, 0x6a, 0x6a,
    0x6a, 0x6a, 0x6a, 0x03, 0x93, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  };
  u8 start_cmd_reply[RX_BUFFER_SIZE];

  rc = sar_init(vpi,vci);

  if (rc < 0)
    goto err_sar_init;


  rc = atm_cell_header_init(vpi,vci,start_cmd,5);

  if (rc < 0) 
    goto err_start_cmd_init;

  start_cmd[3] = 0x0A;
  start_cmd[4] = atm_hec(start_cmd,4);

  printf("Sending the start command...");
  fflush(stdout);
  
  rc = usb_bulk_write(dev_handle, MODEM_DATA_OUT_ENDPOINT,start_cmd,sizeof(start_cmd),TX_TIMEOUT);
  if (rc >= 0) {
    printf("Success\n");
  } else {
    printf("Failed [%s]\n",usb_strerror());
    goto err_write_start_cmd;
  }

  printf("Waiting for reply...");
  fflush(stdout);

  rc = usb_bulk_read(dev_handle, MODEM_DATA_IN_ENDPOINT,start_cmd_reply,sizeof(start_cmd_reply),RX_TIMEOUT);
  if (rc >= 0) {
    printf("Success\n");
  } else {
    printf("Failed [%s]\n",usb_strerror());
    goto err_read_start_cmd_reply;
  }
 

  printf("Starting tx thread...");
  fflush(stdout);
  rc = pthread_create(&pppd_to_modem_thread, NULL, pppd_to_modem, NULL);
  if (rc == 0) {
    printf("Success\n");
  } else {
    printf("Failed [%s]\n",strerror(errno));
    goto err_create_pppd_to_modem;
  }


  printf("Starting rx thread...");
  fflush(stdout);
  rc = pthread_create(&modem_to_pppd_thread, NULL, modem_to_pppd, NULL);
  if (rc == 0) {
    printf("Success\n");
  } else {
    printf("Failed\n");
    goto err_create_modem_to_pppd;
  }
  
  return 0;

 err_create_modem_to_pppd:
  pthread_cancel(pppd_to_modem_thread);
 err_create_pppd_to_modem:
 err_read_start_cmd_reply:
 err_write_start_cmd:
 err_start_cmd_init:
 err_sar_init:
  return rc;
}


static void usage(void)
{
  printf("Usage: adsl-up -vpi <vpi> -vci <vci> [pppd options]\n");
}



static void sig_handler(int signum)
{

  int rc;

  printf("\n\nCaught signal. Exiting\n");

  printf("Killing pppd...");
  fflush(stdout);

  rc = kill(pppd_pid,SIGTERM);
  

  if (rc == 0) {
    printf ("Success\n");
  } else {
    printf("Failed\n");
  }
  


  printf("Cancelling threads\n");

  printf("line status reader...");
  fflush(stdout);

  rc = pthread_cancel(line_status_reader_thread);
  if (rc == 0) {
    printf ("Success\n");
  } else {
    printf("Failed\n");
  }
  
  printf("tx...");
  fflush(stdout);

  rc = pthread_cancel(modem_to_pppd_thread);
  if (rc == 0) {
    printf ("Success\n");
  } else {
    printf("Failed\n");
  }

  printf("rx...");
  fflush(stdout);
  
  rc = pthread_cancel(pppd_to_modem_thread);
  if (rc == 0) {
    printf ("Success\n");
  } else {
    printf("Failed\n");
  }
  
  printf("Releasing interface...\n");
  fflush(stdout);

  rc = usb_release_interface(dev_handle,ifc_num);

  if (rc >= 0) {
    printf ("Success\n");
  } else {
    printf("Failed\n");
  }

  printf("Closing the device handle...");
  fflush(stdout);
  
  rc = usb_close(dev_handle);

  if (rc >= 0) {
    printf ("Success\n");
  } else {
    printf("Failed\n");
  }

  printf("\n\aPlease unplug the modem.\n");

  _exit(0);
}  



int main(int argc, char* argv[])
{
  int rc;
  int cfg_num;
  int i;
  int vpi;
  int vci;

  if (argc < 5 || 
      strcmp(argv[1],"-vpi") != 0 || (vpi = atoi(argv[2])) <= 0 || vpi > 255 ||
      strcmp(argv[3],"-vci") != 0 || (vci = atoi(argv[4])) <= 0 || vci > 65536) {
    usage();
    goto err_args;
  }

  atomic_set(&adsl_line_status,ADSL_STATUS_DOWN);

  signal(SIGINT,sig_handler);
  signal(SIGTERM,sig_handler);

#ifdef DUMP_BUFFERS
  pthread_mutex_init(&log_fp_mutex,NULL);
  log_fp = fopen(LOG_FILE_NAME,"w");
#endif

  usb_init();
  usb_find_busses();
  usb_find_devices();

  printf("Searching for USB Modem..."); 
  fflush(stdout);

  dev = find_device(APACHE_MODEM_VENDOR_ID,
		    APACHE_MODEM_PRODUCT_ID);

  if (dev != NULL) {
    printf("Success (device = %p).\n",dev);
  } else {
    printf("Failed.\n");
    goto err_find_device;
  }
    
  printf("Opening device..."); 
  fflush(stdout);
  
  dev_handle = usb_open(dev);

  if (dev_handle != NULL) {
    printf("Success (handle = %p).\n",dev_handle);
  }
  else {
    printf("Failed [%s].\n",usb_strerror());
    goto err_open;
  }

  cfg_num = dev->config[0].bConfigurationValue;

  /* set the first configuration */
  printf("Setting configuration #%d ...",cfg_num);
  fflush(stdout);

  rc = usb_set_configuration(dev_handle,cfg_num);

  if (rc >= 0) {
    printf ("Success.\n");
  } else {
    printf("Failed [%s].\n",usb_strerror());
    goto err_set_configuration;
  }
  
  ifc_num = dev->config[0].interface[0].altsetting[0].bInterfaceNumber;

  printf("Claiming interface #%d ...",ifc_num);
  fflush(stdout);


  rc = usb_claim_interface(dev_handle,ifc_num);
 
  if (rc >= 0) {
    printf ("Success.\n");
  } else {
    printf("Failed [%s].\n",usb_strerror());
    goto err_claim_interface;
  }


  rc = reset_pipes(dev,dev_handle);
  
  if (rc < 0) {
    goto err_reset_pipes;
  }
  
  printf("Transfering firmware...\n");

  rc = transfer_data(dev_handle,firmware_transfer,FIRMWARE_TRANSFER_SIZE);

  if (rc >= 0) {
    printf("Firmware transfer succeeded.\n");
  } else {
    printf("Firmware transfer failed.\n");
    goto err_transfer_firmware;
  }

  printf("Re-resetting endpoint 0x01... ");
  fflush(stdout);
  rc = usb_resetep(dev_handle,0x01);
  if (rc >= 0) {
    printf("Success.\n");
  } else {
    printf("Failed [%s].\n",usb_strerror());
    goto err_re_reset;
  }

  printf("Starting line status reader thread on endpoint 0x%02x...",MODEM_STATUS_ENDPOINT);
  fflush(stdout);

  rc = pthread_create(&line_status_reader_thread,NULL,line_status_reader,dev_handle);

  if (rc == 0) {
    printf("Success\n");
  } else {
    printf("Failed\n");
    goto err_create_line_status_reader;
  }

  sleep(1);

  rc = transfer_data(dev_handle,setup_transfer,SETUP_TRANSFER_SIZE);
  if (rc >= 0) {
    printf("Modem setup succeeded.\n");
  } else {
    printf("Modem  setup failed.\n");
    goto err_modem_setup;
  }

  printf("Waiting for the line to go up\n");

  for(;;) {
    if (atomic_read(&adsl_line_status) == ADSL_STATUS_UP)
      break;
    rc = usb_bulk_write(dev_handle, line_status_transfer->endpoint, 
			line_status_transfer->data,line_status_transfer->size, TX_TIMEOUT);
    if (rc <  0) {
      printf("Querying modem  status failed.\n");
      goto err_query_modem_before_pppd;
    }

    sleep(1);
  }

  if (atomic_read(&adsl_line_status) != ADSL_STATUS_UP) {
    printf("\nADSL line won't go up. Dying...\n");
    goto err_line_not_up;
  }

  printf("\nADSL line is up. Forking pppd...");
  fflush(stdout);

  /* Set up the pppd command line */
  argc -= 3;
  argv += 3;
  argv[0] = PPPD_EXEC;
  argv[1] = "notty"; /* This option is mandatory */

  rc = start_pppd(argc,argv,&from_pppd_fd,&to_pppd_fd);
  
  if (rc < 0) {
    printf("Failed\n");
    goto err_fork_pppd;
  }
  
  printf("Success\n");
  
  printf("Starting pppd I/O threads...");
  rc = start_pppd_io_threads(vpi,vci);
  
  if (rc < 0) {
    printf("Failed.\n");
    goto err_start_pppd_io_threads;
  }

  printf("Success.\n");
  
  for(;;) {
    rc = usb_bulk_write(dev_handle, line_status_transfer->endpoint, 
			line_status_transfer->data,line_status_transfer->size, TX_TIMEOUT);
    if (rc <  0) {
      printf("Querying modem  status failed.\n");
      goto err_query_modem_after_pppd;
    }

    sleep(1);
  }


  return 0;

 err_query_modem_after_pppd:
 err_start_pppd_io_threads:
  kill(pppd_pid,SIGTERM);
 err_fork_pppd:
 err_line_not_up:
 err_query_modem_before_pppd:
  pthread_cancel(line_status_reader_thread);
 err_create_line_status_reader:
 err_modem_setup:
 err_re_reset:
 err_transfer_firmware:
 err_reset_pipes:
  usb_release_interface(dev_handle,ifc_num);
 err_claim_interface:
 err_set_configuration:
  usb_close(dev_handle);
 err_open:
 err_find_device:
#ifdef DUMP_BUFFERS
  fclose(log_fp);
 err_open_log:
#endif
 err_args:
  return 1;
}
